#
# Copyright (c) 2006 SUSE LINUX Products GmbH, Nuernberg, Germany.
#


package SUSE::SuseRegister;

=head1 NAME

SUSE::SuseRegister - Functions for registration at the Novell registration server

To test for an error you have to look at the context error code $ctx->{errorcode}.
If the errorcode is not equal 0 an error happens. You can get a message via $ctx->{errormsg}.

=cut


use strict;
use XML::Parser;
use XML::Writer;
use Data::Dumper;
use Getopt::Long;
use Encode;
use Sys::Syslog;
use URI;
use URI::QueryParam;
use Time::HiRes qw(gettimeofday tv_interval);
use SUSE::SRPrivate;

=head2 Public Functions

The set of public functions

 use SUSE::SuseRegister;

=over 2

=item *
B<$ctx = init_ctx($data)>

Initialize the context. This function must be called first, before using the 
rest of this functions.

I<$data> is a hash reference and supports the following keys:

* products (array reference of product names)

* nooptional (bool)

* nohwdata (bool)

* forcereg (bool)

* xmlout (bool)

* nozypp (bool)

* dumpfilehack (filename)

* dumpxmlfilehack (filename)

* logfile (filename)

* locale (string)

* noproxy (bool)

* yastcall (bool)

* mirrorCount (integer default is 1)

* debug (bool)

* args (hash reference with 'key' => 'value')

* extraCurlOption (array reference)

* time (bool)

For registration some arguments must be provided to the registration server.
These arguments are stored in side the context $ctx. 
Here is the "$ctx->{args}" definition:

 $ctx->{args} => {
                   '<key>' => {
                                'value' => '<value>',
                                'flag'  => '<i|a|m>',
                                'kind'  => '<m|o>'
                              },
                    ...
                 };

 flag: i == initial   (still send to the server; cannot be discarded)
       a == automatic (automatic collected by the system; maybe discarded - see also "kind")
       m == manual    (must/should be set by the user - see also "kind")

 kind: m == mandatory (value must be set)
       o == optional  (value could be set)

 Every value given via init_ctx has the flag "i" and kind "m"
 After init_ctx only initial arguments are stored in this field. Every "needinfo" 
 during the registration request( I<register($ctx)> ) can modify the arguments 
 in this field.

EXAMPLE:

  my $data = {};
  $data->{nooptional} = 1;
  $data->{noproxy} = 1;

  my $ctx = SUSE::SuseRegister::init_ctx($data);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut

sub init_ctx
{
    my $data = shift;
    my $ctx = {};
    my $code = 0;
    my $msg = "";

    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";
    $ctx->{time} = 0;
    $ctx->{timeindent} = 0;
    if(exists $data->{time} && defined $data->{time})
    {
        $ctx->{time} = $data->{time};
    }
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: init_ctx\n" if($ctx->{time});



    $ctx->{version} = "1.0";

    $ctx->{configFile}      = "/etc/suseRegister.conf";
    $ctx->{sysconfigFile}   = "/etc/sysconfig/suse_register";

    $ctx->{GUID_FILE}       = "/etc/zmd/deviceid";
    $ctx->{SECRET_FILE}     = "/etc/zmd/secret";
    $ctx->{SYSCONFIG_CLOCK} = "/etc/sysconfig/clock";
    $ctx->{CA_PATH}         = "/etc/ssl/certs";

    $ctx->{URL}             = "https://secure-www.novell.com/center/regsvc/";

    $ctx->{URLlistParams}   = "command=listparams";
    $ctx->{URLregister}     = "command=register";
    $ctx->{URLlistProducts} = "command=listproducts";

    $ctx->{guid}      = undef;
    $ctx->{locale}    = undef;
    $ctx->{encoding}  = "utf-8";
    $ctx->{lang}      = "en-US";

    $ctx->{listParams}      = 0;
    $ctx->{xmlout}          = 0;  
    $ctx->{nozypp}          = 0;
    $ctx->{norug}           = 0;
    $ctx->{nozypper}        = 0;
    $ctx->{dumpfilehack}    = ""; # FIXME: is this nedded ?
    $ctx->{dumpxmlfilehack} = ""; # FIXME: is this nedded ?
    $ctx->{nooptional}  = 0;
    $ctx->{acceptmand}  = 0;
    $ctx->{forcereg}    = 0;
    $ctx->{nohwdata}    = 0;
    $ctx->{batch}       = 0;
    $ctx->{logfile}     = undef;
    $ctx->{noproxy}     = 0;
    $ctx->{debug}       = 0;
    $ctx->{yastcall}    = 0; # FIXME: is this still needed?
    $ctx->{LOGDESCR}    = undef;
    $ctx->{args}      = {
                         processor => { flag => "i", value => undef, kind => "mandatory" },
                         platform  => { flag => "i", value => undef, kind => "mandatory" },
                         timezone  => { flag => "i", value => undef, kind => "mandatory" },
                        };
    $ctx->{comlineProducts}     = [];
    $ctx->{serverKnownProducts} = [];
    $ctx->{installedProducts}   = [];
    $ctx->{products} = [];

    $ctx->{installedPatterns} = [];

    $ctx->{extraCurlOption} = [];

    $ctx->{hostGUID} = undef;

    $ctx->{zmdConfig} = {};
    $ctx->{ostarget} = "";

    $ctx->{redirects} = 0;
    $ctx->{registerManuallyURL} = "";
    $ctx->{registerReadableText} = [];
    $ctx->{registerPrivPol} = "";

    $ctx->{querypool}     = "/usr/lib/zypp/zypp-query-pool";
    $ctx->{suserelease}   = "/usr/lib/suseRegister/bin/suse_release";
    $ctx->{rug}           = "/usr/bin/rug";
    $ctx->{zypper}        = "/usr/bin/zypper";
    $ctx->{lsb_release}   = "/usr/bin/lsb_release";
    $ctx->{uname}         = "/bin/uname";
    $ctx->{hwinfo}        = "/usr/sbin/hwinfo";
    $ctx->{zmdInit}       = "/etc/init.d/novell-zmd";
    $ctx->{curl}          = "/usr/bin/curl";

    $ctx->{xenstoreread}  = "/usr/bin/xenstore-read";
    $ctx->{xenstorewrite} = "/usr/bin/xenstore-write";
    $ctx->{xenstorechmod} = "/usr/bin/xenstore-chmod";

    $ctx->{createGuid}    = "/usr/bin/uuidgen";

    $ctx->{lastResponse}    = "";
    $ctx->{initialDomain}   = "";

    $ctx->{rugzmdInstalled} = 0;
    $ctx->{zypperInstalled} = 0;
    
    $ctx->{addRegSrvSrc} = 1;
    $ctx->{addAdSrc} = [];

    $ctx->{mirrorCount} = 1;
    $ctx->{zmdcache} = "/var/cache/SuseRegister/lastzmdconfig.cache";
    

    if(exists $data->{products} && ref($data->{products}) eq "ARRAY")
    {
        $ctx->{comlineProducts} = $data->{products};
    }
    
    if(exists $data->{xmlout} && defined $data->{xmlout})
    {
        $ctx->{xmlout} = $data->{xmlout};
    }

    if(exists $data->{nozypp} && defined $data->{nozypp})
    {
        $ctx->{nozypp} = $data->{nozypp};
    }

    if(exists $data->{norug} && defined $data->{norug})
    {
        $ctx->{norug} = $data->{norug};
    }

    if(exists $data->{nozypper} && defined $data->{nozypper})
    {
        $ctx->{nozypper} = $data->{nozypper};
    }

    if(exists $data->{dumpfilehack} && defined $data->{dumpfilehack})
    {
        $ctx->{dumpfilehack} = $data->{dumpfilehack};
    }

    if(exists $data->{dumpxmlfilehack} && defined $data->{dumpxmlfilehack})
    {
        $ctx->{dumpxmlfilehack} = $data->{dumpxmlfilehack};
    }

    if(exists $data->{nooptional} && defined $data->{nooptional})
    {
        $ctx->{nooptional} = $data->{nooptional};
    }

    if(exists $data->{forcereg} && defined $data->{forcereg})
    {
        $ctx->{forcereg} = $data->{forcereg};
    }

    if(exists $data->{nohwdata} && defined $data->{nohwdata})
    {
        $ctx->{nohwdata} = $data->{nohwdata};
    }

    if(exists $data->{batch} && defined $data->{batch})
    {
        $ctx->{batch} = $data->{batch};
    }

    if(exists $data->{logfile} && defined $data->{logfile})
    {
        $ctx->{logfile} = $data->{logfile};
    }

    if(exists $data->{locale} && defined $data->{locale})
    {
        $ctx->{locale} = $data->{locale};
    }

    if(exists $data->{noproxy} && defined $data->{noproxy})
    {
        $ctx->{noproxy} = $data->{noproxy};
    }

    if(exists $data->{yastcall} && defined $data->{yastcall})
    {
        $ctx->{yastcall} = $data->{yastcall};
    }

    if(exists $data->{mirrorCount} && defined $data->{mirrorCount})
    {
        $ctx->{mirrorCount} = $data->{mirrorCount};
    }

    if(exists $data->{debug} && defined $data->{debug})
    {
        $ctx->{debug} = $data->{debug};
    }

    if(exists $data->{args} && ref($data->{args}) eq "HASH")
    {
        foreach my $a (keys %{$data->{args}})
        {
            $ctx->{args}->{$a} = {flag => "i", value => $data->{args}->{$a}, kind => "mandatory"};
        }
    }

    if(exists $data->{extraCurlOption} && ref($data->{extraCurlOption}) eq "ARRAY")
    {
        $ctx->{extraCurlOption} = $data->{extraCurlOption};
    }

    openlog("suse_register", "ndelay,pid", 'user');

    if(-e $ctx->{zmdInit} && -e $ctx->{rug}) 
    {
        $ctx->{rugzmdInstalled} = 1;
    }
    
    if(-e $ctx->{zypper}) 
    {
        $ctx->{zypperInstalled} = 1;
    }

    # check and fix mirrorCount
    
    if(!defined $ctx->{mirrorCount} || $ctx->{mirrorCount} < 1)
    {
        $ctx->{mirrorCount} = 1;
    }

    
    if(! -e $ctx->{querypool})
    {
        # Code10 compat
        $ctx->{querypool} = "/usr/lib/zmd/query-pool";

        # lib64 hack
        if(! -e $ctx->{querypool}) 
        {
            $ctx->{querypool} = "/usr/lib64/zmd/query-pool";
        
            if(!-e $ctx->{querypool})
            {
                # SLES9 ? try to find suse_release
                if(-e $ctx->{suserelease})
                {
                    $ctx->{querypool} = undef;
                }
                else
                {
                    SUSE::SRPrivate::logPrintError($ctx, "query-pool command not found.\n", 12);
                    return $ctx;
                }
            }
        }
    }
    
    # check for xen tools
    if(! -e $ctx->{xenstoreread} &&
       -e "/bin/xenstore-read")
    {
        $ctx->{xenstoreread} = "/bin/xenstore-read";
    }
    if(! -e $ctx->{xenstorewrite} &&
       -e "/bin/xenstore-write" )
    {
        $ctx->{xenstorewrite} = "/bin/xenstore-write";
    }
    if(! -e $ctx->{xenstorechmod} &&
       -e "/bin/xenstore-chmod" )
    {
        $ctx->{xenstorechmod} = "/bin/xenstore-chmod";
    }
    

    # call this as soon as possible.
    ($code, $msg) = SUSE::SRPrivate::rugStart($ctx);
    if($code != 0)
    {
        SUSE::SRPrivate::logPrintError($ctx, $msg, $code);
        return $ctx;
    }

    if (defined $ctx->{logfile} && $ctx->{logfile} ne "")
    {
        open($ctx->{LOGDESCR}, ">> ".$ctx->{logfile}) or do 
        {
            if(!$ctx->{yastcall})
            {
                SUSE::SRPrivate::logPrintError($ctx, "Cannot open logfile <$ctx->{logfile}>: $!\n", 12);
                return $ctx;
            }
            else
            {
                syslog("err", "Cannot open logfile <$ctx->{logfile}>: $!(yastcall ignoring error)");
                $ctx->{LOGDESCR} = undef;
            }
        };
        # $LOGDESCR is undef if no logfile is defined
        if(defined $ctx->{LOGDESCR})
        {
            print {$ctx->{LOGDESCR}} "----- ".localtime()." ---------------------------------------\n";
        }
    }

    #$ENV{'PATH'} = '/bin:/usr/bin:/sbin:/usr/sbin:/opt/kde3/bin:/opt/gnome/bin';

    my @x = ();
    foreach my $p (@{$ctx->{comlineProducts}})
    {
        push @x, [$p, "0"];
    }
    $ctx->{comlineProducts} = \@x;

    ($code, $msg) = SUSE::SRPrivate::readSystemValues($ctx);
    if($code != 0)
    {
        SUSE::SRPrivate::logPrintError($ctx, $msg, $code);
        return $ctx;
    }


    print STDERR "list-parameters:   $ctx->{listParams}\n" if($ctx->{debug});
    print STDERR "product:           ".Data::Dumper->Dump([$ctx->{comlineProducts}])."\n" if($ctx->{debug});
    print STDERR "xml-output:        $ctx->{xmlout}\n" if($ctx->{debug});
    print STDERR "dumpfile:          $ctx->{dumpfilehack}\n" if($ctx->{debug});
    print STDERR "dumpxmlfile:       $ctx->{dumpxmlfilehack}\n" if($ctx->{debug});
    print STDERR "no-optional:       $ctx->{nooptional}\n" if($ctx->{debug});
    print STDERR "batch:             $ctx->{batch}\n" if($ctx->{debug});
    print STDERR "forcereg:          $ctx->{forcereg}\n" if($ctx->{debug});
    print STDERR "no-hw-data:        $ctx->{nohwdata}\n" if($ctx->{debug});
    print STDERR "norug:             $ctx->{norug}\n" if($ctx->{debug});
    print STDERR "nozypper:          $ctx->{nozypper}\n" if($ctx->{debug});
    print STDERR "log:               ".(($ctx->{logfile})?$ctx->{logfile}:"undef")."\n" if($ctx->{debug});
    print STDERR "locale:            ".(($ctx->{locale})?$ctx->{locale}:"undef")."\n" if($ctx->{debug});
    print STDERR "no-proxy:          $ctx->{noproxy}\n" if($ctx->{debug});
    print STDERR "yastcall:          $ctx->{yastcall}\n" if($ctx->{debug});
    print STDERR "mirrorCount:       $ctx->{mirrorCount}\n" if($ctx->{debug});
    print STDERR "arg: ".Data::Dumper->Dump([$ctx->{args}])."\n" if($ctx->{debug});
    print STDERR "extra-curl-option:".Data::Dumper->Dump([$ctx->{extraCurlOption}])."\n" if($ctx->{debug});
    
    print STDERR "URL:               $ctx->{URL}\n" if($ctx->{debug});
    print STDERR "listParams:        $ctx->{URLlistParams}\n" if($ctx->{debug});
    print STDERR "register:          $ctx->{URLregister}\n" if($ctx->{debug});
    
    
    my $iuri = URI->new($ctx->{URL});

    $ctx->{initialDomain} = $iuri->host;
    $ctx->{initialDomain} =~ s/.+(\.[^.]+\.[^.]+)$/$1/;

    print STDERR "initialDomain:     $ctx->{initialDomain}\n" if($ctx->{debug});

    if(exists $ENV{LANG} && $ENV{LANG} =~ /^([\w_]+)\.?/) 
    {
        if(defined $1 && $1 ne "") 
        {
            $ctx->{lang} = $1;
            $ctx->{lang} =~ s/_/-/;
        }
    }
    elsif(exists $ENV{LANGUAGE} && $ENV{LANGUAGE} =~ /^([\w_]+)\.?/) 
    {
        if(defined $1 && $1 ne "") 
        {
            $ctx->{lang} = $1;
            $ctx->{lang} =~ s/_/-/;
        }
    }

    if (defined $ctx->{locale})
    {
        my ($l, $e) = split(/\.|@/, $ctx->{locale}, 2);
        
        if (defined $l && $l ne "")
        {
            $l =~ s/_/-/;
            $ctx->{lang} = $l;
        }
        
        if (defined $e && $e ne "") 
        {        
            $ctx->{encoding} = $e;
        }
    }

    print STDERR "lang:              $ctx->{lang}\n" if($ctx->{debug});

# set LANG to en_US to get the error messages in english
$ENV{LANG}     = "en_US";
$ENV{LANGUAGE} = "en_US";


    ($code, $msg) = SUSE::SRPrivate::listProducts($ctx);
    if($code != 0)
    {
        SUSE::SRPrivate::logPrintError($ctx, $msg, $code);
        return $ctx;
    }
    
    if(@{$ctx->{comlineProducts}} > 0)
    {
        $ctx->{products} = SUSE::SRPrivate::intersection($ctx, $ctx->{comlineProducts}, $ctx->{installedProducts});
        $ctx->{products} = SUSE::SRPrivate::intersection($ctx, $ctx->{products}, $ctx->{serverKnownProducts});
    }
    else
    {
        $ctx->{products} = SUSE::SRPrivate::intersection($ctx, $ctx->{installedProducts}, $ctx->{serverKnownProducts});
    }

    if($#{$ctx->{installedProducts}} == 0 && 
       exists $ctx->{installedProducts}->[0]->[0] &&
       $ctx->{installedProducts}->[0]->[0] =~ /FACTORY/i)
    {
        SUSE::SRPrivate::logPrintError($ctx, "FACTORY cannot be registered\n", 101);
        return $ctx;
    }

    if(@{$ctx->{products}} == 0)
    {
        SUSE::SRPrivate::logPrintError($ctx, "None of the installed products can be registered at the Novell registration server.\n", 100);
        return $ctx;
    }

    print STDERR SUSE::SRPrivate::indent($ctx)."END: init_ctx:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return $ctx;
}

=item *
B<del_ctx($ctx)>

Delete the context

=cut

sub del_ctx
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: del_ctx\n" if($ctx->{time});
    
    close $ctx->{LOGDESCR} if(defined $ctx->{LOGDESCR});
    closelog;

    $ctx = {};

    print STDERR SUSE::SRPrivate::indent($ctx)."END: del_ctx:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;
}


=item *
B<$txt = listParams($ctx)>

Return the parameter list from the server.

EXAMPLE:

  my $txt = listParams($ctx);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }
  print $txt;

=cut

sub listParams
{
    my $ctx = shift;
    my $text = "";

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: listParams\n" if($ctx->{time});

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    my $output = '<?xml version="1.0" encoding="utf-8"?>';
    
    my $writer = new XML::Writer(OUTPUT => \$output);

    $writer->startTag("listparams", "xmlns" => "http://www.novell.com/xml/center/regsvc-1_0",
                                    "client_version" => "$SUSE::SRPrivate::SRversion");

    foreach my $PArray (@{$ctx->{products}})
    {
        if(defined $PArray->[0] && $PArray->[0] ne "" &&
           defined $PArray->[1] && $PArray->[1] ne "")
        {
            $writer->startTag("product",
                              "version" => $PArray->[1],
                              "release" => $PArray->[2],
                              "arch"    => $PArray->[3]);
            if ($PArray->[0] =~ /\s+/)
            {
                $writer->cdata($PArray->[0]);
            }
            else
            {
                $writer->characters($PArray->[0]);
            }
            $writer->endTag("product");
        }
    }
    
    $writer->endTag("listparams");

    print STDERR "XML:\n$output\n" if($ctx->{debug} >= 3);

    $ctx->{redirects} = 0;                           
                                      # hard coded en-US; suse_register is in english
    my $res = SUSE::SRPrivate::sendData($ctx, $ctx->{URL}."?".$ctx->{URLlistParams}."&lang=en-US&version=$ctx->{version}", $output);
    if($ctx->{errorcode} != 0)
    {
        return;
    }
        
    if ($ctx->{xmlout})
    {
        return "$res\n";
    }
    else
    {
        my $privpol = "";
        
        if(!$ctx->{yastcall})
        {
            $text .= "Available Options:\n\n";
            $text .= "You can add these options to suse_register with the -a option.\n";
            $text .= "'-a' can be used multiple times\n\n";
        }
        
        my $p = new XML::Parser(Style => 'Objects', Pkg => 'SR');
        my $tree = $p->parse($res);

        #print Data::Dumper->Dump([$tree])."\n";

        if (! defined $tree || ref($tree->[0]) ne "SR::paramlist" ||
            ! exists $tree->[0]->{Kids} || ref($tree->[0]->{Kids}) ne "ARRAY")
        {
            SUSE::SRPrivate::logPrintError($ctx, "Invalid XML format. Cannot show human readable output. Try --xml-output.\n".
                                     6);
            return;
        }

        if($ctx->{yastcall})
        {
            $text .= "<pre>";
        }
                
        foreach my $kid (@{$tree->[0]->{Kids}})
        {
            #print Data::Dumper->Dump([$tree->[1]->[$i]])."\n\n";

            if (ref($kid) eq "SR::param")
            {
                if (exists $kid->{command} && defined $kid->{command} &&
                    $ctx->{nohwdata} && $kid->{command} =~ /^hwinfo/)
                {
                    # skip; --no-hw-data was provided 
                    next;
                }
                elsif (exists $kid->{command} && defined $kid->{command} &&
                    $ctx->{nohwdata} && $kid->{command} =~ /^installed-desktops$/)
                {
                    # skip; --no-hw-data was provided 
                    next;
                }
                
                $text .= "* ".$kid->{description}.": ";
                if(!$ctx->{yastcall})
                {
                    $text .= "\n\t".$kid->{id}."=<value> ";
                }

                if (exists $kid->{command} && defined $kid->{command} && $kid->{command} ne "")
                {
                    $text .= "(command: ".$kid->{command}.")\n";
                }
                else
                {
                    $text .= "\n";
                }
                if(!$ctx->{yastcall})
                {
                    $text .= "\n";
                }
            }
            elsif (ref($kid) eq "SR::privacy" )
            {
                if (exists $kid->{description} && defined $kid->{description})
                {
                    if(!$ctx->{yastcall}) 
                    {
                        $privpol .= "\nInformation on Novell's Privacy Policy:\n";
                        $privpol .= $kid->{description}."\n";
                    }
                    else
                    {
                        $privpol .= "<p>Information on Novell's Privacy Policy:<br>\n";
                        $privpol .= $kid->{description}."</p>\n";
                    }
                }
                
                if (exists $kid->{url} && defined $kid->{url} && $kid->{url} ne "")
                {
                    if(!$ctx->{yastcall}) 
                    {
                        $privpol .= $kid->{url}."\n";
                    }
                    else
                    {
                        $privpol .= "<p><a href=\"".$kid->{url}."\">";
                        $privpol .= $kid->{url}."</a></p>\n";
                    }
                }
            }
        }
        if(!$ctx->{yastcall})
        {
            $text .= "Example:\n";
            $text .= "  suse_register -a email=\"tux\@example.com\"\n";
        }
        else
        {
            $text .= "</pre>\n";
        }

        $text .= $privpol;
    }

    print STDERR SUSE::SRPrivate::indent($ctx)."END: listParams:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return "$text\n";
}

=item *
B<$ret = register($ctx)>

Perform the registration.

I<$ret> returns the status of the registration. 

 1 indicates a "needinfo" from the server. 

 0 registration completed

 2 error - but it is better to test $ctx->{errorcode}

Important values in the context are:

 $ctx->{args} the arguments which will be send to the server - see I<init_ctx()> for details.

 $ctx->{registerManuallyURL} contains a URL to a webpage where you can enter the manual values.

 $ctx->{registerReadableText} contains a human readable text about the missing arguments.


EXAMPLE:

  my $ret = register($ctx);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }

  if($ret == 1)
  {
     # you can show and modify the $ctx->{args} here.
     # Have a look for mandatory args which cannot be automaticaly
     # detected. Ask for them, or open a browser with the URL proided
     # in $ctx->{registerManuallyURL} .

  }
  if($ret == 0)
  {
     # Registration completed. You can now get a tasklist to get
     # more information on the update sources (I<getTaskList($ctx)>). 
    
  }

=cut


sub register
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: register\n" if($ctx->{time});


    my $code = 0;
    my $msg = "";
    
    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    my $output = SUSE::SRPrivate::buildXML($ctx);

    $ctx->{redirects} = 0;
    my $res = SUSE::SRPrivate::sendData($ctx, $ctx->{URL}."?".$ctx->{URLregister}."&lang=en-US&version=$ctx->{version}", $output);
    if($ctx->{errorcode} != 0)
    {
        return 2;
    }

    if($res eq $ctx->{lastResponse})
    {
        # Got the same server response as the last time
        SUSE::SRPrivate::logPrintError($ctx, "Invalid response from the registration server. Aborting.\n", 9);
        return 2;
    }
    $ctx->{lastResponse} = $res;

    my $p = new XML::Parser(Style => 'Objects', Pkg => 'SR');
    my $tree = $p->parse($res);
    
    #print Data::Dumper->Dump([$tree])."\n";


    if (defined $tree && ref($tree->[0]) eq "SR::needinfo" && 
        exists $tree->[0]->{Kids} && ref($tree->[0]->{Kids}) eq "ARRAY") 
    {
        if ($ctx->{xmlout})
        {
            $ctx->{xmloutput} = "$res\n";
            return 1;
        }

        # cleanup old stuff
        $ctx->{registerReadableText} = [];
        $ctx->{registerManuallyURL} = "";
        $ctx->{registerPrivPol} = "";

        if (exists $tree->[0]->{href} &&
            defined $tree->[0]->{href} &&
            $tree->[0]->{href} ne "")
        {
            my $uri = URI->new($tree->[0]->{href});
            my $h = $uri->query_form_hash();
            $h->{lang} = "$ctx->{lang}";
            $uri->query_form_hash($h);
            $ctx->{registerManuallyURL} = $uri->as_string;
        }
        
        my $ret = SUSE::SRPrivate::evalNeedinfo($ctx, $tree);
        if($ctx->{errorcode} != 0)
        {
            return 2;
        }
        
        if ($#{$ctx->{registerReadableText}} > -1 && 
            $ctx->{registerReadableText}->[$#{$ctx->{registerReadableText}}] =~ /^\s*$/) 
        {
            pop @{$ctx->{registerReadableText}};
        }
        
        if(!$ctx->{yastcall})
        {
            unshift @{$ctx->{registerReadableText}},
            "To complete the registration, provide some additional parameters:\n\n";
            
            push @{$ctx->{registerReadableText}}, "\nYou can provide these parameters with the '-a' option.\n";
            push @{$ctx->{registerReadableText}}, "You can use the '-a' option multiple times.\n\n";
            push @{$ctx->{registerReadableText}}, "Example:\n\n";
            push @{$ctx->{registerReadableText}}, 'suse_register -a email="me@example.com"'."\n";
            push @{$ctx->{registerReadableText}}, "\nTo register your product manually, use the following URL:\n\n";
            push @{$ctx->{registerReadableText}}, "$ctx->{registerManuallyURL}\n\n";
            
        }
        else 
        {
            unshift @{$ctx->{registerReadableText}}, "<pre>";
            push @{$ctx->{registerReadableText}}, "</pre>";
            push @{$ctx->{registerReadableText}}, "<p>To register your product manually, use the following URL:</p>\n";
            push @{$ctx->{registerReadableText}}, "<pre>$ctx->{registerManuallyURL}</pre>\n\n";
        }
        
        push @{$ctx->{registerReadableText}}, $ctx->{registerPrivPol};
        
        
        # after the first needinfo, set accept=mandatory to true
        # If the application think, that this is not a good idea
        # it can reset this.
        $ctx->{acceptmand} = 1;
        
        print STDERR SUSE::SRPrivate::indent($ctx)."END: register(needinfo):".(tv_interval($t0))."\n" if($ctx->{time});
        $ctx->{timeindent}--;
        
        # return 1 == needinfo
        return 1;
    }
    elsif (defined $tree && ref($tree->[0]) eq "SR::zmdconfig" &&
           exists $tree->[0]->{Kids} && ref($tree->[0]->{Kids}) eq "ARRAY")
    {
        # we ignore possible errors here
        SUSE::SRPrivate::readLastZmdConfig($ctx);
        #SUSE::SRPrivate::saveLastZmdConfig($ctx, $res);
        $ctx->{newzmdconfig} = "$res";

        if ($ctx->{xmlout})
        {
            $ctx->{xmloutput} = "$res\n";
            return 0;
        }

        # HACK: write <zmdconfig> xml for yast
        if(defined $ctx->{dumpxmlfilehack} && $ctx->{dumpxmlfilehack} ne "")
        {
            my $exc = open(XMLD, "> $ctx->{dumpxmlfilehack}") and do 
            {
                print XMLD $res."\n";
                syslog("info", "dumpxmlfile writing:$res");
                close XMLD;
            };
            
            if(!defined $exc || $exc == 0)
            {
                syslog("err", "Cannot write to $ctx->{dumpxmlfilehack}: $!");
            }
        }
        
        # ok => configure zmd
        #configureZMD($ctx, $tree->[0]->{Kids});
        ($code, $msg) = SUSE::SRPrivate::getZmdConfigValues($ctx, $tree->[0]->{Kids});
        if($code != 0)
        {
            SUSE::SRPrivate::logPrintError($ctx, $msg, $code);
            return 2;
        }
    }
    else
    {
        SUSE::SRPrivate::logPrintError($ctx, "Unknown reponse format.\n", 11);
        return 2;
    }
    print STDERR SUSE::SRPrivate::indent($ctx)."END: register(zmdconfig):".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;
    return 0;
}

=item *
B<configureZMD($ctx)>

Executes the update source configuration procedure. This method ignores all error which might 
happen. When you test the context error code after executing this function you can see only
the last error or - if the errorcode is 0 - that no error happens.

If you want another behaviour fetch the tasklist (I<getTaskList()>) and develop your own function 
to configure the sources.

EXAMPLE:

  configureZMD($ctx);
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }
  exit 0; #finished

=cut

sub configureZMD
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: configureZMD\n" if($ctx->{time});

    my $lastcode = 0;
    my $lastmsg = "";

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    my $taskList = getTaskList($ctx);
    if($ctx->{errorcode} != 0)
    {
        return undef;
    }
    
    print Data::Dumper->Dump([$taskList])."\n" if($ctx->{debug} >= 2);

    # search for sources of type NU, RCE or Zenwork, which might need a refresh
    foreach my $service (keys %{$taskList})
    {
        if(lc($ctx->{currentSources}->{$service}->{type}) eq "nu" ||
           lc($ctx->{currentSources}->{$service}->{type}) eq "rce" ||
           lc($ctx->{currentSources}->{$service}->{type}) eq "zenworks")
        {
            # found sources which needs to be refreshed
            SUSE::SRPrivate::rugRefresh($ctx);
            last;
        }
    }
    
    foreach my $service (keys %{$taskList})
    {
        if($taskList->{$service}->{task} eq "a")
        {
            # add this service
            addService($ctx, $taskList->{$service});
            if($ctx->{errorcode} != 0)
            {
                $lastcode = $ctx->{errorcode};
                $lastmsg  = $ctx->{errormsg};
                $ctx->{errorcode} = 0;
                $ctx->{errormsg} = "";
            }
        }
        elsif($taskList->{$service}->{task} eq "d")
        {
            # delete this service
            deleteService($ctx, $taskList->{$service});
            if($ctx->{errorcode} != 0)
            {
                $lastcode = $ctx->{errorcode};
                $lastmsg  = $ctx->{errormsg};
                $ctx->{errorcode} = 0;
                $ctx->{errormsg} = "";
            }
        }
        elsif($taskList->{$service}->{task} eq "le" &&
              (lc($taskList->{$service}->{type}) eq "nu" ||
               lc($taskList->{$service}->{type}) eq "rce" ||
               lc($taskList->{$service}->{type}) eq "zenworks") &&
              exists $taskList->{$service}->{catalog} &&
              ref($taskList->{$service}->{catalog}) eq "HASH")
        {
            # check for changes in the catalogs
            foreach my $name (keys %{$taskList->{$service}->{catalog}})
            {
                if($taskList->{$service}->{catalog}->{$name}->{task} eq "a")
                {
                    # add catalog
                    addCatalog($ctx, $name, $taskList->{$service}->{catalog}->{$name});
                    if($ctx->{errorcode} != 0)
                    {
                        $lastcode = $ctx->{errorcode};
                        $lastmsg  = $ctx->{errormsg};
                        $ctx->{errorcode} = 0;
                        $ctx->{errormsg} = "";
                    }
                }
                elsif($taskList->{$service}->{catalog}->{$name}->{task} eq "d")
                {
                    # delete catalog
                    deleteCatalog($ctx, $name, $taskList->{$service}->{catalog}->{$name});
                    if($ctx->{errorcode} != 0)
                    {
                        if(exists $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name} &&
                           ! exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name} &&
                           ! exists $ctx->{currentSources}->{$service}->{catalog}->{$name})
                        {
                            # maybe the catalog is really not there; we do not set an error here
                            $ctx->{errorcode} = 0;
                            $ctx->{errormsg} = "";
                        }
                        else
                        {
                            $lastcode = $ctx->{errorcode};
                            $lastmsg  = $ctx->{errormsg};
                            $ctx->{errorcode} = 0;
                            $ctx->{errormsg} = "";
                        }
                    }
                }
                elsif($taskList->{$service}->{catalog}->{$name}->{task} eq "r")
                {
                    # re-init catalog
                    reinitCatalog($ctx, $name, $taskList->{$service}->{catalog}->{$name});
                    if($ctx->{errorcode} != 0)
                    {
                        $lastcode = $ctx->{errorcode};
                        $lastmsg  = $ctx->{errormsg};
                        $ctx->{errorcode} = 0;
                        $ctx->{errormsg} = "";
                    }
                }
                # else do nothing
            }
        }
        # else do nothing
    }

    if($lastcode == 0)
    {
        # zmd successfully configured - save the cache file
        SUSE::SRPrivate::saveLastZmdConfig($ctx);
    }

    SUSE::SRPrivate::logPrintError($ctx, $lastmsg, $lastcode);
    
    print STDERR SUSE::SRPrivate::indent($ctx)."END: configureZMD:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return undef;
}



=item *
B<$taskList = getTaskList($ctx)>

Evaluate whichs task should be done.

In the tasklist you can see all sources which exists in the past, current and will be available in the future
with a suggestion what to do with it. Therefor we have some task flags:

 * a  == add this source
 * d  == delete this source
 * r  == re-init (used for catalogs - delete and add it again)
 * le == leave untouched (source is enabled)
 * ld == leave untouched (source is disabled)

The structure of this tasklist looks like this:

$taskList = {
          'ftp://download.example.com/SLES-10-GMC/i386/DVD1?alias=SUSE-Linux-Enterprise-Server' => {
                                'url' => 'ftp://download.example.com/SLES-10-GMC/i386/DVD1?alias=SUSE-Linux-Enterprise-Server',
                                'task' => 'le',
                                'type' => 'ZYPP',
                                'alias' => 'SUSE-Linux-Enterprise-Server'
                                                                                                   },
          'https://update.novell.com' => {
                                'catalog' => {
                                        'SLES10-Updates' => {
                                             'oldurl' => 'https://update.novell.com/repo/$RCE/SLES10-Updates/sles-10-i686/',
                                             'url' => 'https://update.novell.com/repo/$RCE/SLES10-Updates/sles-10-i586/',
                                             'task' => 'r'
                                                            },
                                        'SLE10-Debuginfo-Updates' => {
                                             'url' => 'https://update.novell.com/repo/$RCE/SLE10-Debuginfo-Updates/sles-10-i586/',
                                             'task' => 'le'
                                                                     }
                                              },
                                'url' => 'https://update.novell.com',
                                'task' => 'le',
                                'type' => 'nu',
                                'alias' => 'update_novell_com'
                                        }
           };

=cut

sub getTaskList
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: getTaskList\n" if($ctx->{time});

    my $taskList = {};
    my @allServices = ();
    my $service = undef;
    my $code = 0;
    my $msg = "";
    
    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    print STDERR "getTaskList called\n" if($ctx->{debug} >= 3);


    # call currentServices if they are not set in the context
    if(!exists $ctx->{currentSources} || 
       ref($ctx->{currentSources}) ne "HASH")
    {
        # to be able to run testcases we cannot do this always
        ($code, $msg) = SUSE::SRPrivate::currentServices($ctx);
        if($code != 0)
        {
            SUSE::SRPrivate::logPrintError($ctx, $msg, $code);
            return undef;
        }
    }
    
    print STDERR "---------------- lastZmdConfig --------------------\n".
    Data::Dumper->Dump([$ctx->{lastZmdConfig}])."\n" if($ctx->{debug} >= 3);
    print STDERR "---------------- zmdConfig --------------------\n".
    Data::Dumper->Dump([$ctx->{zmdConfig}])."\n" if($ctx->{debug} >= 3);
    print STDERR "---------------- currentSources --------------------\n".
    Data::Dumper->Dump([$ctx->{currentSources}])."\n" if($ctx->{debug} >= 3);

    
    push @allServices, (keys %{$ctx->{lastZmdConfig}});
    push @allServices, (keys %{$ctx->{zmdConfig}});
    push @allServices, (keys %{$ctx->{currentSources}});

    # unique
    @allServices = keys %{{map {$_,1} @allServices}};


    # check all cases of the services

    foreach $service (@allServices)
    {
        # $services exists only in lastZmdConfig
        if(exists $ctx->{lastZmdConfig}->{$service} &&
           ! exists $ctx->{zmdConfig}->{$service} &&
           ! exists $ctx->{currentSources}->{$service})
        {
            # nothing to do
            print STDERR "$service exists only in lastZmdConfig: do nothing\n" if($ctx->{debug} >= 3);
        }
        # $services exists only in zmdConfig
        elsif(! exists $ctx->{lastZmdConfig}->{$service} &&
              exists $ctx->{zmdConfig}->{$service} &&
              ! exists $ctx->{currentSources}->{$service})
        {
            # add it
            print STDERR "$service exists only in zmdConfig: add it\n" if($ctx->{debug} >= 3);
            $taskList->{$service} = {};
            SUSE::SRPrivate::copyService($taskList->{$service}, $ctx->{zmdConfig}->{$service});
            $taskList->{$service}->{task} = "a";
        }
        # $services exists only in currentSources
        elsif(! exists $ctx->{lastZmdConfig}->{$service} &&
              ! exists $ctx->{zmdConfig}->{$service} &&
              exists $ctx->{currentSources}->{$service})
        {
            # leave enabled
            print STDERR "$service exists only in currentSource: leave enabled\n" if($ctx->{debug} >= 3);
            $taskList->{$service} = {};
            SUSE::SRPrivate::copyService($taskList->{$service}, $ctx->{currentSources}->{$service});
            $taskList->{$service}->{task} = "le";
        }
        # $services exists only in lastZmdConfig and zmdConfig
        elsif(exists $ctx->{lastZmdConfig}->{$service} &&
              exists $ctx->{zmdConfig}->{$service} &&
              ! exists $ctx->{currentSources}->{$service})
        {
            # leave disabled
            print STDERR "$service exists in lastZmdConfig and zmdConfig: leave disabled\n" if($ctx->{debug} >= 3);
            $taskList->{$service} = {};
            SUSE::SRPrivate::copyService($taskList->{$service}, $ctx->{zmdConfig}->{$service});
            $taskList->{$service}->{task} = "ld";
        }
        # $services exists only in lastZmdConfig and currentSources
        elsif(exists $ctx->{lastZmdConfig}->{$service} &&
              ! exists $ctx->{zmdConfig}->{$service} &&
              exists $ctx->{currentSources}->{$service})
        {
            # delete it
            print STDERR "$service exists in lastZmdConfig and currentSOurce: delete it\n" if($ctx->{debug} >= 3);
            $taskList->{$service} = {};
            SUSE::SRPrivate::copyService($taskList->{$service}, $ctx->{lastZmdConfig}->{$service});
            $taskList->{$service}->{task} = "d";
        }
        # $services exists only in zmdConfig and currentSources
        elsif(! exists $ctx->{lastZmdConfig}->{$service} &&
              exists $ctx->{zmdConfig}->{$service} &&
              exists $ctx->{currentSources}->{$service})
        {
            # leave enabled
            print STDERR "$service exists in zmdConfig and currentSource: leave enabled\n" if($ctx->{debug} >= 3);
            $taskList->{$service} = {};
            SUSE::SRPrivate::copyService($taskList->{$service}, $ctx->{zmdConfig}->{$service});
            $taskList->{$service}->{task} = "le";
        }
        # $services exists in all
        elsif(exists $ctx->{lastZmdConfig}->{$service} &&
              exists $ctx->{zmdConfig}->{$service} &&
              exists $ctx->{currentSources}->{$service})
        {
            # leave enabled
            print STDERR "$service exists in all: leave enabled\n" if($ctx->{debug} >= 3);
            $taskList->{$service} = {};
            SUSE::SRPrivate::copyService($taskList->{$service}, $ctx->{zmdConfig}->{$service});
            $taskList->{$service}->{task} = "le";
        }
    }
    
    # check catalogs for not yum/zypp service types

    foreach $service (keys %{$taskList})
    {
        print STDERR "search for special types in service '$service'\n" if($ctx->{debug} >= 3);

        # action only on "le" all other types have the correct catalog values
        #
        # "a" - it is not in currentSources and the catalog values are from zmdConfig
        # "d" - we want to delete it and the catalog values are from lastZmdConfig
        # "ld" - it is disabled and the catalog values are from zmdConfig
        
        if((lc($taskList->{$service}->{type}) eq "nu" ||
            lc($taskList->{$service}->{type}) eq "rce" ||
            lc($taskList->{$service}->{type}) eq "zenworks") &&
           exists $taskList->{$service}->{catalog} &&
           $taskList->{$service}->{task} eq "le")
        {
            print STDERR "Found type with catalogs and status le\n" if($ctx->{debug} >= 3);
            
            if(! exists $ctx->{lastZmdConfig}->{$service} &&
               ! exists $ctx->{zmdConfig}->{$service} &&
               exists $ctx->{currentSources}->{$service})
            {
                # do nothing
                print STDERR "catalog only in currentSources - do nothing\n" if($ctx->{debug} >= 3);

                # copy the subscribed tag to task 
                foreach my $name (keys %{$taskList->{$service}->{catalog}})
                {
                    my $task = ($taskList->{$service}->{catalog}->{$name}->{subscribed} == 1)?"le":"ld";
                    $taskList->{$service}->{catalog}->{$name}->{task} = $task;
                }
                
            }
            elsif(! exists $ctx->{lastZmdConfig}->{$service} &&
                  exists $ctx->{zmdConfig}->{$service} &&
                  exists $ctx->{currentSources}->{$service})
            {
                # add or delete catalogs is possible here
                print STDERR "catalog in zmdConfig and currentSources\n" if($ctx->{debug} >= 3);
                
                my @allCatalogNames = ();
                push @allCatalogNames, (keys %{$ctx->{zmdConfig}->{$service}->{catalog}});
                push @allCatalogNames, (keys %{$ctx->{currentSources}->{$service}->{catalog}});
                
                # unique
                @allCatalogNames = keys %{{map {$_,1} @allCatalogNames}};

                # first we delete the catalog entry of the current $taskList
                # and add an empty one
                #delete $taskList->{$service}->{catalog};
                $taskList->{$service}->{catalog} = {};

                # go through all cases
                foreach my $name (@allCatalogNames)
                {
                    print STDERR "check catalog '$name'\n" if($ctx->{debug} >= 3);

                    # catalog name exists only in zmdConfig
                    if(exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name} &&
                       ! exists $ctx->{currentSources}->{$service}->{catalog}->{$name})
                    {
                        print STDERR "1 add it\n" if($ctx->{debug} >= 3);

                        # add it
                        foreach my $key (keys %{$ctx->{zmdConfig}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{zmdConfig}->{$service}->{catalog}->{$name}->{$key};
                        }
                        $taskList->{$service}->{catalog}->{$name}->{task} = "a";
                    }
                    # catalog name exists only in currentSources
                    elsif(! exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name} &&
                          exists $ctx->{currentSources}->{$service}->{catalog}->{$name})
                    {
                        print STDERR "2 delete it\n" if($ctx->{debug} >= 3);

                        # delete it - !!! PROBLEM - We do not have the URL of the catalog for YaST !!!
                        foreach my $key (keys %{$ctx->{currentSources}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{currentSources}->{$service}->{catalog}->{$name}->{$key};
                        }
                        $taskList->{$service}->{catalog}->{$name}->{task} = "d";
                    }
                    else     # both
                    {
                        print STDERR "3 leave enabled\n" if($ctx->{debug} >= 3);

                        # exists in both leave enabled
                        foreach my $key (keys %{$ctx->{zmdConfig}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{zmdConfig}->{$service}->{catalog}->{$name}->{$key};
                        }

                        # check subscribed
                        if(exists $ctx->{currentSources}->{$service}->{catalog}->{$name}->{subscribed} &&
                           $ctx->{currentSources}->{$service}->{catalog}->{$name}->{subscribed} == 1)
                        {
                            $taskList->{$service}->{catalog}->{$name}->{task} = "le";
                        }
                        else
                        {
                            $taskList->{$service}->{catalog}->{$name}->{task} = "a";
                        }
                    }
                }
            }
            else  # $service is in all sources
            {
                # add, delete and re-init is possible

                print STDERR "catalog in all sources\n" if($ctx->{debug} >= 3);

                my @allCatalogNames = ();
                push @allCatalogNames, (keys %{$ctx->{lastZmdConfig}->{$service}->{catalog}});
                push @allCatalogNames, (keys %{$ctx->{zmdConfig}->{$service}->{catalog}});
                push @allCatalogNames, (keys %{$ctx->{currentSources}->{$service}->{catalog}});
                
                print STDERR "unique catalog count: ".$#allCatalogNames."\n" if($ctx->{debug} >= 3);

                # unique
                @allCatalogNames = keys %{{map {$_,1} @allCatalogNames}};
                
                print STDERR "unique catalog count: ".$#allCatalogNames."\n" if($ctx->{debug} >= 3);
                

                # first we delete the catalog entry of the current $taskList
                # and add an empty one
                #delete $taskList->{$service}->{catalog};
                $taskList->{$service}->{catalog} = {};

                # go through all cases
                foreach my $name (@allCatalogNames)
                {
                    print STDERR "check catalog '$name'\n" if($ctx->{debug} >= 3);

                    # catalog name exists only in lastZmdConfig
                    if(exists $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name} &&
                       ! exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name} &&
                       ! exists $ctx->{currentSources}->{$service}->{catalog}->{$name})
                    {
                        print STDERR "case 1 delete it\n" if($ctx->{debug} >= 3);

                        # delete it - maybe it is still in YaST
                        foreach my $key (keys %{$ctx->{lastZmdConfig}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name}->{$key};
                        }
                        $taskList->{$service}->{catalog}->{$name}->{task} = "d";
                    }
                    # catalog name exists only in zmdConfig
                    elsif(! exists $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name} &&
                          exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name} &&
                          ! exists $ctx->{currentSources}->{$service}->{catalog}->{$name})
                    {
                        print STDERR "case 2 add it\n" if($ctx->{debug} >= 3);

                        # add it
                        foreach my $key (keys %{$ctx->{zmdConfig}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{zmdConfig}->{$service}->{catalog}->{$name}->{$key};
                        }
                        $taskList->{$service}->{catalog}->{$name}->{task} = "a";
                    }
                    # catalog name exists only in currentSources
                    elsif(! exists $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name} &&
                          ! exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name} &&
                          exists $ctx->{currentSources}->{$service}->{catalog}->{$name})
                    {
                        print STDERR "case 3 leave enabled\n" if($ctx->{debug} >= 3);

                        # leave enabled
                        foreach my $key (keys %{$ctx->{currentSources}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{currentSources}->{$service}->{catalog}->{$name}->{$key};
                        }
                        $taskList->{$service}->{catalog}->{$name}->{task} = "le";
                    }
                    # catalog name exists only in lastZmdConfig and zmdConfig
                    elsif(exists $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name} &&
                          exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name} &&
                          ! exists $ctx->{currentSources}->{$service}->{catalog}->{$name})
                    {
                        print STDERR "case 4 leave disabled\n" if($ctx->{debug} >= 3);

                        # leave disabled
                        foreach my $key (keys %{$ctx->{zmdConfig}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{zmdConfig}->{$service}->{catalog}->{$name}->{$key};
                        }
                        $taskList->{$service}->{catalog}->{$name}->{task} = "ld";
                    }
                    # catalog name exists only in lastZmdConfig and currentSources
                    elsif(exists $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name} &&
                          ! exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name} &&
                          exists $ctx->{currentSources}->{$service}->{catalog}->{$name})
                    {
                        print STDERR "case 5 delete it\n" if($ctx->{debug} >= 3);

                        # delete it
                        foreach my $key (keys %{$ctx->{lastZmdConfig}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name}->{$key};
                        }
                        $taskList->{$service}->{catalog}->{$name}->{task} = "d";
                    }
                    # catalog name exists only in zmdConfig and currentSources
                    elsif(!exists $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name} &&
                          exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name} &&
                          exists $ctx->{currentSources}->{$service}->{catalog}->{$name})
                    {
                        foreach my $key (keys %{$ctx->{zmdConfig}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{zmdConfig}->{$service}->{catalog}->{$name}->{$key};
                        }

                        # check for subscribed in currentSources
                        if(exists $ctx->{currentSources}->{$service}->{catalog}->{$name}->{subscribed} &&
                           $ctx->{currentSources}->{$service}->{catalog}->{$name}->{subscribed} == 1)
                        {
                            print STDERR "case 6a leave enabled\n" if($ctx->{debug} >= 3);

                            # still subscribed - leave enabled
                            $taskList->{$service}->{catalog}->{$name}->{task} = "le";
                        }
                        else
                        {
                            print STDERR "case 6b add it\n" if($ctx->{debug} >= 3);

                            # add it
                            $taskList->{$service}->{catalog}->{$name}->{task} = "a";
                        }
                    }
                    else     # exists in all 
                    {
                        # exists in all 
                        foreach my $key (keys %{$ctx->{zmdConfig}->{$service}->{catalog}->{$name}})
                        {
                            $taskList->{$service}->{catalog}->{$name}->{$key} = $ctx->{zmdConfig}->{$service}->{catalog}->{$name}->{$key};
                        }

                        # check for changes in the URL
                        if(exists $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name}->{url} &&
                           exists $ctx->{zmdConfig}->{$service}->{catalog}->{$name}->{url} &&
                           $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name}->{url} ne $ctx->{zmdConfig}->{$service}->{catalog}->{$name}->{url})
                        {
                            print STDERR "case 7a re-init\n" if($ctx->{debug} >= 3);

                            $taskList->{$service}->{catalog}->{$name}->{task} = "r";
                            $taskList->{$service}->{catalog}->{$name}->{oldurl} = $ctx->{lastZmdConfig}->{$service}->{catalog}->{$name}->{url};
                        }
                        else
                        {
                            if(exists $ctx->{currentSources}->{$service}->{catalog}->{$name}->{subscribed} &&
                               $ctx->{currentSources}->{$service}->{catalog}->{$name}->{subscribed} == 1)
                            {
                                print STDERR "case 7b leave enabled\n" if($ctx->{debug} >= 3);

                                # leave enabled
                                $taskList->{$service}->{catalog}->{$name}->{task} = "le";
                            }
                            else
                            {
                                print STDERR "case 7c leave disabled\n" if($ctx->{debug} >= 3);
                        
                                # leave disabled
                                $taskList->{$service}->{catalog}->{$name}->{task} = "ld";
                            }
                        }
                    }
                }
            }
        }
        elsif((lc($taskList->{$service}->{type}) eq "nu" ||
               lc($taskList->{$service}->{type}) eq "rce" ||
               lc($taskList->{$service}->{type}) eq "zenworks") &&
              exists $taskList->{$service}->{catalog})
        {
            # copy the service task to the catalogs
            foreach my $name (keys %{$taskList->{$service}->{catalog}})
            {
                my $task = $taskList->{$service}->{task};
                $taskList->{$service}->{catalog}->{$name}->{task} = $task;
            }
        }
    }

    print STDERR SUSE::SRPrivate::indent($ctx)."END: getTaskList:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;
    
    return $taskList;
}

=item *
B<$ret = addService($ctx, $service)>

Add a Service. Reports only the last error.

EXAMPLE:

 
  my $ret = addService($ctx, $taskList->{http://update.novell.com/});
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }
  # add more services here

 
=cut

sub addService
{
    my $ctx     = shift;
    my $service = shift || undef;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: addService\n" if($ctx->{time});

    my $lastmsg     = "";
    my $lastcode    = 0;
    my $msg     = "";
    my $code    = 0;
    my $url     = undef;
    my $type    = undef;
    my $alias   = undef;
    my $regcode = undef;
    
    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    $url     = $service->{url}     if(exists $service->{url}) ;
    $type    = $service->{type}    if(exists $service->{type});
    $alias   = $service->{alias}   if(exists $service->{alias});
    $regcode = $service->{regcode} if(exists $service->{regcode});

    if (! defined $url || $url eq "") 
    {
        SUSE::SRPrivate::logPrintError($ctx, "Missing URL.\n", 14);
        return 14;
    }
    if (! defined $type || $type eq "") 
    {
        SUSE::SRPrivate::logPrintError($ctx, "Missing Service Type.\n", 14);
        return 14;
    }

    if($ctx->{rugzmdInstalled})
    {
        ($code, $msg) = SUSE::SRPrivate::rugServiceAdd($ctx, $url, $type, $alias, $regcode);
        if($code != 0)
        {
            $lastcode = $code;
            $lastmsg  = $msg;
        }
        
        if((lc($type) eq "nu" ||
            lc($type) eq "rce" ||
            lc($type) eq "zenworks") &&
           exists $service->{catalog} &&
           ref($service->{catalog}) eq "HASH")
        {
            foreach my $name (keys %{$service->{catalog}})
            {
                addCatalog($ctx, $name, $service->{catalog}->{$name});
                if($ctx->{errorcode} != 0)
                {
                    $lastcode = $ctx->{errorcode};
                    $lastmsg  = $ctx->{errormsg};
                    $ctx->{errorcode} = 0;
                    $ctx->{errormsg} = "";
                }
            }
        }
        elsif((lc($type) eq "yum" ||
              lc($type) eq "zypp") && !$ctx->{nozypp})
        {
            # need a subscribe too
            my $cat_name = $url;
            $cat_name    = $alias if (defined $alias && $alias ne "");

            ($code, $msg) = SUSE::SRPrivate::rugCatalogAdd($ctx, $cat_name);
            if ($code != 0)
            {
                $lastcode = $code;
                $lastmsg = $msg;
            }
        }
    }
    else
    {
        if((lc($type) eq "nu" ||
            lc($type) eq "rce" ||
            lc($type) eq "zenworks") &&
           exists $service->{catalog} &&
           ref($service->{catalog}) eq "HASH")
        {
            foreach my $name (keys %{$service->{catalog}})
            {
                addCatalog($ctx, $name, $service->{catalog}->{$name});
                if($ctx->{errorcode} != 0)
                {
                    $lastcode = $ctx->{errorcode};
                    $lastmsg  = $ctx->{errormsg};
                    $ctx->{errorcode} = 0;
                    $ctx->{errormsg} = "";
                }
            }
        }
        elsif(lc($type) eq "zypp" ||
              lc($type) eq "yum")
        {
            ($code, $msg) = SUSE::SRPrivate::zyppServiceAdd($ctx, $url, $type, $alias);
            if($code != 0)
            {
                $lastcode = $code;
                $lastmsg  = $msg;
            }
        }
    }
    SUSE::SRPrivate::logPrintError($ctx, $msg, $code);

    print STDERR SUSE::SRPrivate::indent($ctx)."END: addService:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return $code;
}

=item *
B<$ret = addCatalog($ctx, $name, $catalog)>

Add a catalog. Reports only the last error.

EXAMPLE:

 
  my $ret = addCatalog($ctx, "SLES10-Updates", $taskList->{http://update.novell.com/}->{catalog}->{SLES10-Updates});
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut

sub addCatalog
{
    my $ctx     = shift;
    my $name    = shift || undef;
    my $catalog = shift || undef;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: addCatalog\n" if($ctx->{time});

    my $lastmsg     = "";
    my $lastcode    = 0;
    my $msg     = "";
    my $code    = 0;

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    if(!defined $name || ! defined $catalog || ref($catalog) ne "HASH")
    {
        SUSE::SRPrivate::logPrintError($ctx, "Invalid Catalog", 14);
        return 14;
    }
    
   
    # first add it with zypp if url is available
    if(exists $catalog->{url})
    {
        my $url = SUSE::SRPrivate::fillURL($ctx, $catalog->{url}, undef);
        
        # A rug catalog is a zypp service - so call zyppServiceAdd
        ($code, $msg) = SUSE::SRPrivate::zyppServiceAdd($ctx, $url, "zypp", $name);
        if($code != 0)
        {
            $lastcode = $code;
            $lastmsg = $msg;
        }
    }
    
    ($code, $msg) = SUSE::SRPrivate::rugCatalogAdd($ctx, $name);
    if($code != 0)
    {
        $lastcode = $code;
        $lastmsg = $msg;
    }
    
    SUSE::SRPrivate::logPrintError($ctx, $lastmsg, $lastcode);

    print STDERR SUSE::SRPrivate::indent($ctx)."END: addCatalog:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return 14;
}


=item *
B<$ret = reinitCatalog($ctx, $name, $catalog)>

Re-initialize a catalog. Reports only the last error. Requires the oldurl paramter in the catalog.

EXAMPLE:

 
  my $ret = reinitCatalog($ctx, "SLES10-Updates", $taskList->{http://update.novell.com/}->{catalog}->{SLES10-Updates});
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }

=cut

sub reinitCatalog
{
    my $ctx     = shift;
    my $name    = shift || undef;
    my $catalog = shift || undef;

    my $msg     = "";
    my $code    = 0;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: reinitCatalog\n" if($ctx->{time});


    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    if(!defined $name || ! defined $catalog || ref($catalog) ne "HASH")
    {
        SUSE::SRPrivate::logPrintError($ctx, "Invalid Catalog", 14);
        return 14;
    }

    if(! exists $catalog->{oldurl})
    {
        SUSE::SRPrivate::logPrintError($ctx, "Missing old url", 14);
        return 14;
    }
    
    my $dummyCatalog = { url => $catalog->{oldurl}, task => "d" };
        
    deleteCatalog($ctx, $name, $dummyCatalog);
    if($ctx->{errorcode} != 0)
    {
        $code = $ctx->{errorcode};
        $msg = $ctx->{errormsg};
        $ctx->{errorcode} = 0;
        $ctx->{errormsg} = "";
    }
    
    $dummyCatalog = { url => $catalog->{url}, task => "a" };
    
    addCatalog($ctx, $name, $dummyCatalog);
    if($ctx->{errorcode} != 0)
    {
        $code = $ctx->{errorcode};
        $msg = $ctx->{errormsg};
        $ctx->{errorcode} = 0;
        $ctx->{errormsg} = "";
    }
    
    SUSE::SRPrivate::logPrintError($ctx, $msg, $code);

    print STDERR SUSE::SRPrivate::indent($ctx)."END: reinitCatalog:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return $code;
}


=item *
B<$ret = deleteService($ctx, $service)>

Delete a Service. Reports only the last error.

EXAMPLE:

 
  my $ret = deleteService($ctx, $taskList->{http://update.novell.com/});
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut

sub deleteService
{
    my $ctx = shift;
    my $service = shift || undef;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: deleteService\n" if($ctx->{time});

    my $lastmsg     = "";
    my $lastcode    = 0;

    my $msg     = "";
    my $code    = 0;
    my $url     = undef;
    my $type    = undef;
    my $alias   = undef;
    
    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    $url     = $service->{url}     if(exists $service->{url}) ;
    $type    = $service->{type}    if(exists $service->{type});
    $alias   = $service->{alias}   if(exists $service->{alias});

    if (! defined $url || $url eq "") 
    {
        SUSE::SRPrivate::logPrintError($ctx, "Missing URL.\n", 14);
        return 14;
    }
    if (! defined $type || $type eq "") 
    {
        SUSE::SRPrivate::logPrintError($ctx, "Missing Service Type.\n", 14);
        return 14;
    }

    if($ctx->{rugzmdInstalled})
    {
        if((lc($type) eq "nu" ||
            lc($type) eq "rce" ||
            lc($type) eq "zenworks") &&
           exists $service->{catalog} &&
           ref($service->{catalog}) eq "HASH")
        {
            foreach my $name (keys %{$service->{catalog}})
            {
                deleteCatalog($ctx, $name, $service->{catalog}->{$name});
                if($ctx->{errorcode} != 0)
                {
                    $lastcode = $ctx->{errorcode};
                    $lastmsg = $ctx->{errormsg};
                    $ctx->{errorcode} = 0;
                    $ctx->{errormsg} = "";
                }
            }
        }

        ($code, $msg) = SUSE::SRPrivate::rugServiceDelete($ctx, $url, $type);
        if($code != 0)
        {
            $lastcode = $code;
            $lastmsg = $msg;
        }
    }
    else
    {
        if((lc($type) eq "nu" ||
            lc($type) eq "rce" ||
            lc($type) eq "zenworks") &&
           exists $service->{catalog} &&
           ref($service->{catalog}) eq "HASH")
        {
            foreach my $name (keys %{$service->{catalog}})
            {
                deleteCatalog($ctx, $name, $service->{catalog}->{$name});
                if($ctx->{errorcode} != 0)
                {
                    $lastcode = $ctx->{errorcode};
                    $lastmsg = $ctx->{errormsg};
                    $ctx->{errorcode} = 0;
                    $ctx->{errormsg} = "";
                }
            }
        }
        elsif(lc($type) eq "zypp" ||
              lc($type) eq "yum")
        {
            ($code, $msg) = SUSE::SRPrivate::zyppServiceDelete($ctx, $alias);
            if($code != 0 )
            {
                # try with the URL
                ($code, $msg) = SUSE::SRPrivate::zyppServiceDelete($ctx, $url);
                
                # code 32 is "not found" which can be ignored on delete
                if($code != 0 && $code != 32)
                {
                    $lastcode = $code;
                    $lastmsg = $msg;
                }
            }
        }
    }
    SUSE::SRPrivate::logPrintError($ctx, $lastmsg, $lastcode);

    print STDERR SUSE::SRPrivate::indent($ctx)."END: deleteService:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;
    
    return $lastcode;
}


=item *
B<$ret = deleteCatalog($ctx, $name, $catalog)>

Delete a Catalog. Reports only the last error.

EXAMPLE:

 
  my $ret = deleteCatalog($ctx, "SLES10-Updates", $taskList->{http://update.novell.com/}->{catalog}->{SLES10-Updates});
  if($ctx->{errorcode} != 0)
  {
    print STDERR $ctx->{errormsg}."\n";
    exit $ctx->{errorcode};
  }


=cut

sub deleteCatalog
{
    my $ctx     = shift;
    my $name    = shift || undef;
    my $catalog = shift || undef;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR SUSE::SRPrivate::indent($ctx)."START: deleteCatalog\n" if($ctx->{time});

    my $lastmsg     = "";
    my $lastcode    = 0;
    my $msg     = "";
    my $code    = 0;

    # cleanup the error status
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    if(!defined $name || ! defined $catalog || ref($catalog) ne "HASH")
    {
        return SUSE::SRPrivate::logPrintError($ctx, "Invalid Catalog", 14);
        return 14;
    }
   
    # first delete it with zypp if url is available
    
    # A rug catalog is a zypp service - so call zyppServiceDelete
    ($code, $msg) = SUSE::SRPrivate::zyppServiceDelete($ctx, $name);

    if($code != 0 && exists $catalog->{url})
    {
        # try with the URL
        # A rug catalog is a zypp service - so call zyppServiceDelete
        ($code, $msg) = SUSE::SRPrivate::zyppServiceDelete($ctx, $catalog->{url});
    
        # code 32 is "not found" which can be ignored on delete
        if($code != 0 && $code != 32)
        {
            $lastcode = $code;
            $lastmsg = $msg;
        }
    }
    
    ($code, $msg) = SUSE::SRPrivate::rugCatalogDelete($ctx, $name);
    if($code != 0)
    {
        $lastcode = $code;
        $lastmsg = $msg;
    }
    
    SUSE::SRPrivate::logPrintError($ctx, $lastmsg, $lastcode);

    print STDERR SUSE::SRPrivate::indent($ctx)."END: deleteCatalog:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return $lastcode;
}



#=======================================================================================================
#===== END OF PUBLIC FUNCTIONS =========================================================================
#=======================================================================================================


sub fullpathOf 
{
    my $ctx = shift;
    my $program = shift || undef;
    
    if(!defined $program || $program eq "" || $program !~ /^[\w_-]+$/)
    {
        return undef;
    }
    
    my $fullpath = `which $program 2>/dev/null`;
    chomp($fullpath);
    print STDERR "Fullpath:$fullpath\n" if($ctx->{debug} >=2);
    
    if (defined $fullpath && $fullpath ne "")
    {
        return $fullpath;
    }
    return undef;
}



1;


