#!/bin/bash
#
# create_repo_for_patch.sh
#
# Copyright (C) 2006 Novell Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin Street,
# Fifth Floor,
# Boston, MA  02110-1301,
# USA.
#

MYSTART=`pwd`
PATCHESFILE="patches.xml"
PREFIX="patch-"
DEBUG="no"
SIGN_ID=""

function usage (){
    echo
    echo "usage: $0 [options] <source directory containing RPMs> <target directory>"
    echo "       -h          : help (this message)"
    echo "       -p          : patch files (if not located in <target_directory>)"
    echo "       -P          : prefix for patchfiles (default: $PREFIX)"
    echo "       -S          : detached sign the repomd.xml file"
    echo "       -I <KEY_ID> : key-id to use for signing the repomd.xml file, if not given"
    echo "                     gpg will try to use the default signing key"
    echo "       -v          : be verbose"
    echo
    echo " example: $0 -S $RPMDIR ."
    echo
    echo "WARING: $0 will delete all files except patch files n <target directory> !"
    echo
    exit $1
}

function check_binary(){
    local BINARY=$(which $1 2>/dev/null)
    if [ "$BINARY" != "" ]; then
        if [ -x "$BINARY" ]; then
            echo "$BINARY"
        else
            echo "$BINARY found but is not executable - please fix!" >&2
            exit 1 
        fi
    else
        echo "$1 not found! Please install it first!" >&2
	    exit 1
    fi
}


if [ ! $1 ]; then
    usage 1
fi

while getopts 'hvSI:p:P:'  OPTION ; do
    case $OPTION in
        h) usage 0
        ;;
        p) NEWPATCHFILES=$OPTARG
        ;;
        P) PREFIX=$OPTARG
        ;;
        S) SIGN="yes"
        ;;
        I) SIGN_ID=$OPTARG
        ;;
        v) DEBUG="yes"
        ;;
        *) usage
        ;;
    esac
done

shift $(( OPTIND - 1 ))

RPMDIR=$1
REPODIR=$2

cd $REPODIR
CREATEREPO=$(check_binary "createrepo")
#SIGN=$(check_binary "sign")
SIGN="sign"

if [ "$CREATEREPO" = "" ] || [ "$SIGN" = "" ]; then
	exit 1
fi

if [ -z $RPMDIR ]; then
 echo "ERROR:  source directory not defined" >&2
 usage 1
fi

if [ -z $REPODIR ]; then
 echo "ERROR:  target directory not defined" >&2
 usage 1
fi

if [ -d $REPODIR/rpm ]; then
    rm -r $REPODIR/rpm/*
else
    mkdir $REPODIR/rpm
fi

if [ -d $REPODIR/repodata.tmp ]; then
    echo "ERROR:  $REPODIR/repodata.tmp exists - exiting" >&2
    exit 1
fi       

mkdir -p $REPODIR/repodata.tmp

if [ -z "$NEWPATCHFILES" ]; then
    for i in `find -name "${NEWPATCHFILES}*"`; do
        mv $i $REPODIR/repodata.tmp/
        PATCHFILES="$PATCHFILES `basename $i`"
    done
else
    for i in $NEWPATCHFILES; do
        find -name $i -exec mv {} $REPODIR/repodata.tmp/
        PATCHFILES="$PATCHFILES `basename $i`"
    done
fi

test -d $REPODIR/repodata && rm -r $REPODIR/repodata
test -d $REPODIR/.olddata && rm -r $REPODIR/.olddata
mkdir $REPODIR/repodata
 
echo "INFO:   --- Copying RPMs from $RPMDIR"

mkdir -p $REPODIR/rpm/src/

for patchfile in `ls $REPODIR/repodata.tmp/*`; do
	patch=`basename $patchfile`
	for rpmfile in `grep "location href" $REPODIR/$patchfile  | cut -d "\"" -f2`; do
		arch_dir=`dirname $rpmfile | xargs basename`
		test -d $REPODIR/rpm/$arch_dir || mkdir -p $REPODIR/rpm/$arch_dir
		if [[ $rpmfile == *.delta.rpm ]]; then continue; fi
		if [[ $rpmfile == *.patch.rpm ]]; then continue; fi
		cp -a $RPMDIR/$rpmfile $REPODIR/rpm/$arch_dir/ 
		if [ $? -ne 0 ]; then
			echo "ERROR:  $RPMDIR/$rpmfile                        missing!" >&2
		else 
			if [ "$DEBUG" == "yes" ]; then
				echo "INFO:   `basename $rpmfile`"
			fi
		fi
		SRCRPM=`rpm -qp --qf "%{SOURCERPM}" $RPMDIR/$rpmfile`
		cp -a $RPMDIR/rpm/src/$SRCRPM $REPODIR/rpm/src/
		if [ $? -ne 0 ]; then
			echo "ERROR:  $RPMDIR/rpm/src/$SRCRPM    missing!"  >&2
		else
                        if [ "$DEBUG" == "yes" ]; then
                                echo "INFO:   $SRCRPM"
                        fi
		fi
	done
done

#
# createrepo
#
echo "INFO:   --- Creating repodata"
if [ "$DEBUG" == "yes" ]; then
  $CREATEREPO -x "*.delta.rpm" -x "*.patch.rpm" $REPODIR
else
  $CREATEREPO -x "*.delta.rpm" -x "*.patch.rpm" $REPODIR 2>/dev/null 1>&2
fi

#
# patches.xml
#
echo "INFO:   --- Creating new file $REPODIR/repodata/$PATCHESFILE"
echo "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" > $REPODIR/repodata/$PATCHESFILE 
echo "<patches xmlns=\"http://novell.com/package/metadata/suse/patches\">" >> $REPODIR/repodata/$PATCHESFILE 
for patch in `ls $REPODIR/repodata.tmp/patch*`; do
	patchname=`basename $patch`
    patchid=`basename $patchname .xml`
    sha1=`sha1sum $patch | awk '" " { print $1}'`
    mv $REPODIR/repodata.tmp/$patchname $REPODIR/repodata/ || next;
	echo "  <patch id=\"$patchid\">" >> $REPODIR/repodata/$PATCHESFILE
	echo "    <checksum type=\"sha\">$sha1</checksum>" >> $REPODIR/repodata/$PATCHESFILE
    echo "    <location href=\"repodata/$patchname\"/>" >> $REPODIR/repodata/$PATCHESFILE
	echo "  </patch>" >> $REPODIR/repodata/$PATCHESFILE
done
echo "</patches>" >> $REPODIR/repodata/$PATCHESFILE

#
# repomd.xml
#
echo "INFO:   --- patching $REPODIR/repodata/repomd.xml"
sha1=`sha1sum $REPODIR/repodata/$PATCHESFILE | awk '" " { print $1}'`
timestamp=`date +"%Y%m%d%H"`
echo "  <data type=\"patches\">" > $REPODIR/repodata.tmp/repomd.xml.patch$$
echo "    <location href=\"repodata/patches.xml\"/>" >> $REPODIR/repodata.tmp/repomd.xml.patch$$
echo "    <checksum type=\"sha\">$sha1</checksum>" >> $REPODIR/repodata.tmp/repomd.xml.patch$$ 
echo "    <timestamp>$timestamp</timestamp>" >> $REPODIR/repodata.tmp/repomd.xml.patch$$ 
echo "    <open-checksum type=\"sha\">$sha1</open-checksum>" >>  $REPODIR/repodata.tmp/repomd.xml.patch$$ 
echo "  </data>" >> $REPODIR/repodata.tmp/repomd.xml.patch$$
echo "</repomd>" >> $REPODIR/repodata.tmp/repomd.xml.patch$$
perl -pi -e 's#</repomd>\n##' $REPODIR/repodata/repomd.xml
cat $REPODIR/repodata.tmp/repomd.xml.patch$$ >> $REPODIR/repodata/repomd.xml
rm -r $REPODIR/repodata.tmp

#
# repomd.xml.key
#
if [ "$SIGN" == "yes" ]; then
	echo "INFO:   --- Signing repomd.xml"
	if [ "$SIGN_ID" != "" ]; then
		gpg -a -b --default-key $SIGN_ID $REPODIR/repodata/repomd.xml
	else
		if [ -f /etc/sign.conf ]; then
		  if [ "$DEBUG" == "yes" ]; then
		     echo "INFO:   Try to sign with SUSE build key"
		  fi
		  $SIGN -d $REPODIR/repodata/repomd.xml
		else
		  gpg -a -b $REPODIR/repodata/repomd.xml
		fi
	  SIGN_ID=`gpg --verify $REPODIR/repodata/repomd.xml.asc 2>&1 | sed -ne "s/.* ID //p"`
	  if [ "$DEBUG" == "yes" ]; then
		echo "INFO:   Signing with: $SIGN_ID"
  	  fi
	fi
        gpg -a --export $SIGN_ID > $REPODIR/repodata/repomd.xml.key
fi

cd $MYSTART

