#! /usr/bin/perl -w
#
# This is the API part of UsersPluginFingerprintReader plugin -
# configuration of user's fingerprints on specific device
#

package UsersPluginFingerprintReader;

use strict;

use YaST::YCP qw(:LOGGING);
use YaPI;
use Data::Dumper;

textdomain("fingerprint-reader");

our %TYPEINFO;

##--------------------------------------
##--------------------- global imports

YaST::YCP::Import ("Directory");
YaST::YCP::Import ("FileUtils");
YaST::YCP::Import ("FingerprintReader");
YaST::YCP::Import ("Pam");
YaST::YCP::Import ("SCR");
YaST::YCP::Import ("Users");

##--------------------------------------
##--------------------- global variables

# error message, returned when some plugin function fails
my $error	= "";

my $fingerprint_reader_available = undef;

# path to store fingerprints
my $bir_path	= "/etc/pam_thinkfinger";

my $name	= "UsersPluginFingerprintReader";
   
##----------------------------------------
##--------------------- internal functions

# internal function:
# check if given key (second parameter) is contained in a list (1st parameter)
# if 3rd parameter is true (>0), ignore case
sub contains {
    my ($list, $key, $ignorecase) = @_;
    if (!defined $list || ref ($list) ne "ARRAY" || @{$list} == 0) {
	return 0;
    }
    if ($ignorecase) {
        if ( grep /^\Q$key\E$/i, @{$list} ) {
            return 1;
        }
    } else {
        if ( grep /^\Q$key\E$/, @{$list} ) {
            return 1;
        }
    }
    return 0;
}

# helper, check if Fingerprint Reader was already configured
sub fingerprint_reader_configured {

    return Pam->Enabled ("thinkfinger");
}

# helper function: check if Fingerprint Reader (the device) is available
sub is_fingerprint_reader_available {

    if (not defined $fingerprint_reader_available) {
	my @devices	= @{FingerprintReader->ReadFingerprintReaderDevices ()};
	$fingerprint_reader_available = (@devices > 0);
    }
    return $fingerprint_reader_available;
}

# check if given user has fingerprint authentication configured
sub fingerprint_present {

    my ($data) = @_;
    my $username	= $data->{"uid"};
    return 0 if !$username;
    my $org_username	= $data->{"org_user"}{"uid"} || $username;
    my $org_bir_file	= "$bir_path/$org_username.bir";
    return (FileUtils->Exists ($org_bir_file));
}

# check if bir file name corresponds with username
sub adapt_fingerprint_info {

    my ($config, $data) = @_;
    my $username	= $data->{"uid"};
    my $org_username	= $data->{"org_user"}{"uid"} || $username;
    my $bir_file	= $data->{"bir_file"};
    return $data if !defined $username;
    if (!defined $bir_file && $username ne $org_username) {
	# When user was only renamed, but new fingerprint was not generated,
	# there's no 'bir_file' entry in the map ->
	# backup his original bir file to tmpdir so it won't get lost and can
	# be correctly renamed in Write.
	my $org_bir_file	= "$bir_path/$org_username.bir";
	if (FileUtils->Exists ($org_bir_file)) {
	    my $tmpdir	= Directory->tmpdir ();
	    SCR->Execute (".target.bash", "/bin/cp $org_bir_file $tmpdir/$username.bir");
	    $data->{"bir_file"}	= $username;
	}
    }
    if ($bir_file && $bir_file ne $username) {
	$data->{"bir_file"}	= $username;
    }
    return $data;
}

# update the object data when removing plugin
sub remove_plugin_data {

    my ($config, $data)		= @_;
    $data->{"bir_file"}		= "";
    $data->{"plugin_modified"}	= 1;
    return $data;
}

##------------------------------------------
##--------------------- global API functions

# All functions have 2 "any" parameters: these mean:
# 1st: configuration map (hash) - e.g. saying if we work with user or group
# 2nd: data map (hash) of user/group to work with
# for details, see UsersPluginLDAPAll.pm

# return names of provided functions
BEGIN { $TYPEINFO{Interface} = ["function", ["list", "string"], "any", "any"];}
sub Interface {

    my $self		= shift;
    my @interface 	= (
	    "GUIClient",
	    "Name",
	    "Summary",
	    "Restriction",
	    "Interface",
	    "PluginPresent",
	    "PluginRemovable",
	    "Error",
	    "AddBefore",
	    "Add",
	    "EditBefore",
	    "Edit",
	    "Delete",
	    "Write"
    );
    return \@interface;
}

# return error message, generated by plugin
BEGIN { $TYPEINFO{Error} = ["function", "string", "any", "any"];}
sub Error {

    my $self		= shift;
    return $error;
}


# return plugin name, used for GUI (translated)
BEGIN { $TYPEINFO{Name} = ["function", "string", "any", "any"];}
sub Name {

    my $self		= shift;
    # plugin name
    return __("Fingerprint configuration");
}

##------------------------------------
# return plugin summary (to be shown in table with all plugins)
BEGIN { $TYPEINFO{Summary} = ["function", "string", "any", "any"];}
sub Summary {

    my $self	= shift;
    # plugin summary (table item)
    my $ret 	= __("Set the user's fingerprint");
    return $ret;
}

##------------------------------------
# checks the current data map of user (2nd parameter) and returns
# true if given user has our plugin
BEGIN { $TYPEINFO{PluginPresent} = ["function", "boolean", "any", "any"];}
sub PluginPresent {

    my ($self, $config, $data)  = @_;
    return 0 if not fingerprint_reader_configured ();

    if (contains ($data->{'plugins'}, $name, 1) || fingerprint_present ($data))
    {
	y2milestone ("$name plugin present");
	return 1;
    }
    return 0;
}

##------------------------------------
# Is it possible to remove this plugin from user?
BEGIN { $TYPEINFO{PluginRemovable} = ["function", "boolean", "any", "any"];}
sub PluginRemovable {
    # doesn't have sense to remove
    return YaST::YCP::Boolean (1);
}


##------------------------------------
# return name of YCP client defining YCP GUI
BEGIN { $TYPEINFO{GUIClient} = ["function", "string", "any", "any"];}
sub GUIClient {

    return "users_plugin_fingerprint_reader";
}

##------------------------------------
# Type of objects this plugin is restricted to.
# It defines:
#	1. type of objects which it should be applied to (ldap/nis/local/system)
#	2. type of objects at all (user)
# If this function doesn't exist, plugin is applied for all users of all types
BEGIN { $TYPEINFO{Restriction} = ["function",
    ["map", "string", "any"], "any", "any"];}
sub Restriction {

    my $self	= shift;
    # do the check here, so the plugin is not shown when there is no hw for it
    return {} if not is_fingerprint_reader_available ();
    return {
	    "local"	=> 1,
	    "system"	=> 1,
	    # only for users
	    "user"	=> 1,
    };
}

# this will be called at the beggining of Users::Add
# Could be called multiple times for one user
BEGIN { $TYPEINFO{AddBefore} = ["function",
    ["map", "string", "any"],
    "any", "any"];
}
sub AddBefore {

    my ($self, $config, $data)  = @_;

    if (!fingerprint_reader_configured ()) {
	# error popup
	$error	= __("Fingerprint reader is not configured. Run 'yast2-fingerprint-reader' to configure it");
	return undef;
    }
    return $data;
}

# This will be called at the end of Users::Add* :
# modify the object map with fingerprint-reader data
BEGIN { $TYPEINFO{Add} = ["function", ["map", "string", "any"], "any", "any"];}
sub Add {

    my ($self, $config, $data)  = @_;
    if (contains ($data->{'plugins_to_remove'}, $name, 1)) {
	y2milestone ("removing plugin $name...");
	return remove_plugin_data ($config, $data);
    }
    return adapt_fingerprint_info ($config, $data);
}

# this will be called at the beggining of Users::Edit
BEGIN { $TYPEINFO{EditBefore} = ["function",
    ["map", "string", "any"],
    "any", "any"];
}
sub EditBefore {

    my ($self, $config, $data)  = @_;

    if (!fingerprint_reader_configured ()) {
	# error popup
	$error	= __("Fingerprint reader is not configured. Run 'yast2-fingerprint-reader' to configure it");
	return undef;
    }
    return $data;
}

# This will be called at the end of Users::Edit* :
# modify the object map with fingerprint-reader data
BEGIN { $TYPEINFO{Edit} = ["function",
    ["map", "string", "any"],
    "any", "any"];
}
sub Edit {

    my ($self, $config, $data)  = @_;
    # "plugins_to_remove" is list of plugins which are set for removal
    if (contains ($data->{'plugins_to_remove'}, $name, 1)) {
	y2milestone ("removing plugin $name...");
	return remove_plugin_data ($config, $data);
    }
    return adapt_fingerprint_info ($config, $data);
}


# What should be done after user is finally written (this is called only once):
# - remove org_username.bir file,
#  but only when there is no other (new) user of that name
# - move temporary bir file to correct location
BEGIN { $TYPEINFO{Write} = ["function", "boolean", "any", "any"];}
sub Write {

    my ($self, $config, $data)  = @_;

    my $username	= $data->{"uid"};
    my $org_username	= $data->{"org_user"}{"uid"} || $username;
    my $bir_file	= $data->{"bir_file"};
    my $org_bir_file	= "$bir_path/$org_username.bir";

    if (($config->{"modified"} || "") eq "deleted") {
	# check if some new user doesn't have the name of the deleted one
	my $u	= Users->GetUserByName ($org_username, "");
	if (!defined $u->{"bir_file"}) {
	    $bir_file	= "";
	}
    }
    return if (!$username || ! defined $bir_file);
    if ($username ne $org_username) {
	# check if there was old bir-file
	my $u	= Users->GetUserByName ($org_username, "");
	# do not delete birfile when some new user was added with the name
	# of this one's original name
	if (!defined $u->{"bir_file"} && FileUtils->Exists ($org_bir_file)) {
	    y2milestone ("removing old file $org_bir_file owned by $username");
	    SCR->Execute (".target.bash", "/bin/rm $org_bir_file");
	}
    }
    elsif ($bir_file eq "" && FileUtils->Exists ($org_bir_file)) {
	# disable fingerprint authentication for this user
	# (now $org_username is same as $username)
	y2milestone ("removing file $org_bir_file for user $username");
	SCR->Execute (".target.bash", "/bin/rm $org_bir_file");
    }
    if ($bir_file) {
	my $bir_file_path	= Directory->tmpdir ();
        $bir_file_path	= "$bir_file_path/$bir_file.bir";
        y2milestone ("moving $bir_file_path to $bir_path/");
	SCR->Execute (".target.bash", "/bin/mv $bir_file_path $bir_path/");
    }
    return YaST::YCP::Boolean (1);
}


42
# EOF
