/**
 * @file librtasevent.h
 * @brief Structure and Interface definitions for librtasevent
 *
 * This provides the structure and interface definitions for librtasevent.
 * Also see librtasevent_v4.h and librtasevent_v6.h for additional
 * structure definitions particular those event versions.
 *
 * librtasevent usage:
 * The librtasevent library is meant to place the structure definitions,
 * parsing and printing of RTAS events into a common place.  The use of
 * librtasevent is easiest through the following interfaces:
 * 
 * parse_rtas_event() - This takes a buffer containing an RTAS event in
 *     binary form and returns a pointer to an rtas_event struct.  This
 *     struct has a list of rtas_event_scn objects hanging off of it
 *     for each section of the rtas event.
 * 
 * This presents the user with a broken down representation of the RTAS
 * event that can then be easily searched for any relevant information
 * or passed to either rtas_print_event() to print the entire RTAS event
 * or the rtas_print_event_scn() to print a particular section of the
 * RTAS event.
 *
 * When finished a call to rtas_cleanup_event() will free all of the data
 * structuires associated with the RTAS event.
 *
 * Copyright (C) 2005 IBM Corporation
 * Common Public License Version 1.0 (see COPYRIGHT)
 *
 * @author Nathan Fontenot <nfont@austin.ibm.com>
 */

#ifndef _H_RTAS_EVENTS
#define _H_RTAS_EVENTS

#include <stdio.h>
#include <inttypes.h>

struct rtas_event;

/**
 * @struct rtas_event_scn
 * @brief Contains the data for a RETAS event section
 */
struct rtas_event_scn {
    struct rtas_event_scn   *next;
    char                    *data;
    int                     scn_id;
    struct rtas_event       *re;
    struct rtas_event_scn   *subscn;
    struct rtas_event_scn   *other_scns;
};

/* definitions for struct rtas_event_scn scn_id */
#define RTAS_EVENT_HDR		1
#define RTAS_EVENT_EXT_HDR	2

#define RTAS_EPOW_SCN		3
#define RTAS_IO_SCN		4

#define RTAS_CPU_SCN		5
#define RTAS_IBM_DIAG_SCN	6	
#define RTAS_MEM_SCN		7
#define RTAS_POST_SCN		8
#define RTAS_IBM_SP_SCN 	9
#define RTAS_VEND_SPECIFIC_SCN  10

#define RTAS_MAIN_A_SCN         11
#define RTAS_MAIN_B_SCN	        12
#define RTAS_DUMP_SCN		13	
#define RTAS_LRI_SCN		14
#define RTAS_MTMS_SCN		15
#define RTAS_PSRC_SCN		16
#define RTAS_SSRC_SCN		17
#define RTAS_SRC_SUBSCN         18
#define RTAS_FRU_SCN            19
#define RTAS_FRU_SUBSCN         20

#define RTAS_MAX_SCN_ID		20 

/**
 * @struct rtas_event
 * @brief Anchor structure for parsed RTAS events
 */
struct rtas_event {
    char    *buffer;
    int     length;
    int     version;
    int     event_no;
    struct rtas_event_scn *event_scns;
};

/**
 * @struct rtas_date
 * @brief definition of date format in rtas events
 */
struct rtas_date {
    uint32_t    year:16;
    uint32_t    month:8;
    uint32_t    day:8;
};

/**
 * @struct rtas_time
 * @brief definition of timestamp in rtas events
 */
struct rtas_time {
    uint32_t    hour:8;
    uint32_t    minutes:8;
    uint32_t    seconds:8;
    uint32_t    hundredths:8;
};

/**
 * @struct rtas_event_hdr
 * @brief Fixed header at the beginning of all RTAS events
 */
struct rtas_event_hdr {
    uint32_t version:8;           /**< Architectural version */
    uint32_t severity:3;          /**< Severity level of error */
#define RTAS_HDR_SEV_NO_ERROR           0
#define RTAS_HDR_SEV_EVENT              1
#define RTAS_HDR_SEV_WARNING            2
#define RTAS_HDR_SEV_ERROR_SYNC         3
#define RTAS_HDR_SEV_ERROR              4
#define RTAS_HDR_SEV_FATAL              5
#define RTAS_HDR_SEV_ALREADY_REPORTED   6

    uint32_t disposition:2;       /**< Degree of recovery */
#define RTAS_HDR_DISP_FULLY_RECOVERED   0
#define RTAS_HDR_DISP_LIMITED_RECOVERY  1
#define RTAS_HDR_DISP_NOT_RECOVERED     2
    
    uint32_t extended:1;          /**< extended log present? */
    uint32_t /* reserved */ :2;
    uint32_t initiator:4;         /**< Initiator of event */
#define RTAS_HDR_INIT_UNKNOWN           0
#define RTAS_HDR_INIT_CPU               1
#define RTAS_HDR_INIT_PCI               2
#define RTAS_HDR_INIT_ISA               3
#define RTAS_HDR_INIT_MEMORY            4
#define RTAS_HDR_INIT_HOT_PLUG          5

    uint32_t target:4;            /**< Target of failed operation */
#define RTAS_HDR_TARGET_UNKNOWN         0
#define RTAS_HDR_TARGET_CPU             1
#define RTAS_HDR_TARGET_PCI             2
#define RTAS_HDR_TARGET_ISA             3
#define RTAS_HDR_TARGET_MEMORY          4
#define RTAS_HDR_TARGET_HOT_PLUG        5

    uint32_t type:8;              /**< General event or error*/
#define RTAS_HDR_TYPE_RETRY             1
#define RTAS_HDR_TYPE_TCE_ERR           2
#define RTAS_HDR_TYPE_INTERN_DEV_FAIL   3
#define RTAS_HDR_TYPE_TIMEOUT           4
#define RTAS_HDR_TYPE_DATA_PARITY       5
#define RTAS_HDR_TYPE_ADDR_PARITY       6
#define RTAS_HDR_TYPE_CACHE_PARITY      7
#define RTAS_HDR_TYPE_ADDR_INVALID      8
#define RTAS_HDR_TYPE_ECC_UNCORRECTED   9
#define RTAS_HDR_TYPE_ECC_CORRECTED     10 
#define RTAS_HDR_TYPE_EPOW              64 
#define RTAS_HDR_TYPE_PLATFORM_ERROR    224
#define RTAS_HDR_TYPE_IBM_IO_EVENT      225
#define RTAS_HDR_TYPE_PLATFORM_INFO     226
#define RTAS_HDR_TYPE_RESOURCE_DEALLOC  227
#define RTAS_HDR_TYPE_DUMP_NOTIFICATION 228

    uint32_t ext_log_length:32;   /**< length in bytes */
};

/**
 * @struct rtas_event_exthdr
 * @brief RTAS optional extended error log header (12 bytes)
 */
struct rtas_event_exthdr {
    uint32_t valid:1;
    uint32_t unrecoverable:1;
    uint32_t recoverable:1;
    uint32_t unrecoverable_bypassed:1;  
    uint32_t predictive:1;
    uint32_t newlog:1;
    uint32_t bigendian:1; 
    uint32_t /* reserved */:1;

    uint32_t platform_specific:1;       /**< only in version 3+ */
    uint32_t /* reserved */:3;
    uint32_t platform_value:4;          /**< valid iff platform_specific */

    uint32_t power_pc:1;               
    uint32_t /* reserved */:2;
    uint32_t addr_invalid:1;     
    uint32_t format_type:4;
#define RTAS_EXTHDR_FMT_CPU             1
#define RTAS_EXTHDR_FMT_MEMORY          2
#define RTAS_EXTHDR_FMT_IO              3
#define RTAS_EXTHDR_FMT_POST            4
#define RTAS_EXTHDR_FMT_EPOW            5
#define RTAS_EXTHDR_FMT_IBM_DIAG        12
#define RTAS_EXTHDR_FMT_IBM_SP          13
#define RTAS_EXTHDR_FMT_VEND_SPECIFIC_1 14
#define RTAS_EXTHDR_FMT_VEND_SPECIFIC_2 15

    /* This group is in version 3+ only */
    uint32_t non_hardware:1;     /**<  Firmware or software is suspect */
    uint32_t hot_plug:1;         
    uint32_t group_failure:1;    
    uint32_t /* reserved */:1;

    uint32_t residual:1;         /**< Residual error from previous boot */
    uint32_t boot:1;             /**< Error during boot */
    uint32_t config_change:1;    /**< Configuration chang since last boot */
    uint32_t post:1;

    struct rtas_time time;       /**< Time of error in BCD HHMMSS00 */
    struct rtas_date date;       /**< Time of error in BCD YYYYMMDD */
};

/* Retrieving and free'ing parsed RTAS events */ 
struct rtas_event * parse_rtas_event(char *, int);
int cleanup_rtas_event(struct rtas_event *);

/* Retrieving a particular section from a parsed RTAS event */
struct rtas_event_scn * rtas_get_event_hdr_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_event_exthdr_scn(struct rtas_event *);

struct rtas_event_scn * rtas_get_epow_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_io_scn(struct rtas_event *);

struct rtas_event_scn * rtas_get_cpu_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_ibmdiag_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_mem_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_post_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_ibm_sp_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_vend_specific_scn(struct rtas_event *);

struct rtas_event_scn * rtas_get_main_a_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_main_b_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_dump_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_lri_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_mtms_scn(struct rtas_event *);
struct rtas_event_scn * rtas_get_src_scn(struct rtas_event *);

/* Printing RTAS event data */
int rtas_print_scn(FILE *, struct rtas_event_scn *, int);
int rtas_print_event(FILE *, struct rtas_event *, int); 
int rtas_print_raw_event(FILE *, struct rtas_event *);
int rtas_set_print_width(int);

/* include version specific data */
#include "librtasevent_v4.h"
#include "librtasevent_v6.h"

#endif
