
package SUSE::SRPrivate;

use strict;
use XML::Parser;
use XML::Writer;
use Data::Dumper;
use File::Temp qw(tempfile);
use File::Copy;
use Sys::Syslog;
use IPC::Open3;
use Fcntl qw(:DEFAULT);
use URI;
use URI::QueryParam;
use Time::HiRes qw(gettimeofday tv_interval);


# client version number
our $SRversion = "1.2.1";

sub readSystemValues
{
    my $ctx = shift;
    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: readSystemValues\n" if($ctx->{time});

    my $code = 0;
    my $msg = "";
    
    ############### batch mode ########################
    
    if($ctx->{batch})
    {
        # if --no-optional or --no-hw-data are not given in batch mode
        # read the sysconfig values for the default

        my $sysconfigOptional = "false";
        my $sysconfigHWData   = "false";
        
        
        open(CNF, "< $ctx->{sysconfigFile}") and do {
        
            while(<CNF>)
            {
                if($_ =~ /^\s*#/)
                {
                    next;
                }
                elsif($_ =~ /^SUBMIT_OPTIONAL\s*=\s*"*([^"\s]*)"*\s*/ && defined $1 && $1 ne "") 
                {
                    $sysconfigOptional = $1;
                    
                }
                elsif($_ =~ /^SUBMIT_HWDATA\s*=\s*"*([^"\s]*)"*\s*/ && defined $1 && $1 ne "") 
                {
                    $sysconfigHWData = $1;
                }
            }
            close CNF;
        };

        if(!$ctx->{nooptional})
        {
            if(lc($sysconfigOptional) eq "true")
            {
                $ctx->{nooptional} = 0;
            }
            else
            {
                $ctx->{nooptional} = 1;
            }
        }
        if(!$ctx->{nohwdata})
        {
            if(lc($sysconfigHWData) eq "true")
            {
                $ctx->{nohwdata} = 0;
            }
            else
            {
                $ctx->{nohwdata} = 1;
            }
        }
   }
        
    ############### read the config ###################
    if(-e $ctx->{configFile})
    {
        open(CNF, "< $ctx->{configFile}") or do 
        {
            return logPrintReturn($ctx, "Cannot open file $ctx->{configFile}: $!\n", 12);
        };
        
        while(<CNF>)
        {
            if($_ =~ /^\s*#/)
            {
                next;
            }
            elsif($_ =~ /^url\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "") 
            {
                $ctx->{URL} = $1;
            }
            elsif($_ =~ /^listParams\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "") 
            {
                $ctx->{URLlistParams} = $1;
            }
            elsif($_ =~ /^listProducts\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "") 
            {
                $ctx->{URLlistProducts} = $1;
            }
            elsif($_ =~ /^register\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "")
            {
                $ctx->{URLregister} = $1;
            }
            elsif($_ =~ /^hostGUID\s*=\s*(\w*)\s*/ && defined $1 && $1 ne "")
            {
                $ctx->{hostGUID} = $1;
            }
            elsif($_ =~ /^addRegSrvSrc\s*=\s*(\w*)\s*/ && defined $1)
            {
                if(lc($1) eq "true")
                { 
                    $ctx->{addRegSrvSrc} = 1;
                }
                else 
                {
                    $ctx->{addRegSrvSrc} = 0;
                }
            }
            elsif($_ =~ /^addAdSrc\s*=\s*(\S*)\s*/ && defined $1 && $1 ne "")
            {
                push @{$ctx->{addAdSrc}}, $1;
            }
        }
        close CNF;
    }
    
    ############### GUID ##############################

    if(!-e $ctx->{GUID_FILE})
    {
        ($code, $msg) = rugStart($ctx);
        if($code != 0)
        {
            return ($code, $msg);
        }
    }
    
    open(ZMD, "< $ctx->{GUID_FILE}") or do 
    {
        return logPrintReturn($ctx, "Cannot open file $ctx->{GUID_FILE}: $!\n", 12);
    };
    
    $ctx->{guid} = <ZMD>;
    chomp($ctx->{guid});
    close ZMD;
    
    print STDERR "GUID:$ctx->{guid}\n" if($ctx->{debug});

    ############### find Products #####################

    ($code, $msg) = getProducts($ctx);
    if($code != 0)
    {
        return ($code, $msg);
    }
    
    ########## host GUID (virtualization) #############

    # spec not finished and this seems not to work

     if(-d "/proc/xen" &&
        -e $ctx->{xenstoreread}) 
     {
         print STDERR "Found XEN\n" if($ctx->{debug} >= 2);
       
         # FIXME: check if this command really returns what we want
         my $val = `$ctx->{xenstoreread} domid 2>/dev/null`;
         $code = ($?>>8);
         chomp($val);
        
         if($code == 0 && defined $val && $val eq "0")
         {
             print STDERR "We are Domain-0\n" if($ctx->{debug} >= 2);
           
             # we are Domain-0    
             if(-e $ctx->{xenstorewrite} && -e $ctx->{xenstorechmod}) 
             {
                 print STDERR "Write /tool/SR/HostDeviceID to xenbus\n" if($ctx->{debug});
               
                 `$ctx->{xenstorewrite} /tool/SR/HostDeviceID $ctx->{guid} 2>/dev/null`;
                 `$ctx->{xenstorechmod} /tool/SR/HostDeviceID r 2>/dev/null`;
             }
         }
         elsif($code == 0 && defined $val && $val ne "")
         {
             print STDERR "try to read /tool/SR/HostDeviceID from xenbus\n" if($ctx->{debug} >= 2);
             
             $val = `$ctx->{xenstoreread} /tool/SR/HostDeviceID 2>/dev/null`;
             chomp($val);
             
             if(defined $val && $val ne "") 
             {
                 print STDERR "Got /tool/SR/HostDeviceID: $val\n" if($ctx->{debug});
               
                 $ctx->{hostGUID} = $val;
             }
         }
         else
         {
             print STDERR "Cannot read from xenstore: $code\n" if($ctx->{debug});
         }
     }

    ############## some initial values ########################

    $ctx->{args}->{processor} = { flag => "i", value => `$ctx->{uname} -p`, kind => "mandatory"};
    $ctx->{args}->{platform}  = { flag => "i", value => `$ctx->{uname} -i`, kind => "mandatory"};
    $ctx->{args}->{timezone}  = { flag => "i", value => "US/Mountain", kind => "mandatory"};      # default


    open(SYSC, "< $ctx->{SYSCONFIG_CLOCK}") or do
    {
        return logPrintReturn($ctx, "Cannot open file $ctx->{SYSCONFIG_CLOCK}: $!\n", 12);
    };
    while(<SYSC>) 
    {
        if($_ =~ /^TIMEZONE\s*=\s*"?([^"]*)"?/) 
        {
            if(defined $1 && $1 ne "")
            {
                $ctx->{args}->{timezone}  = { flag => "i", value => $1, kind => "mandatory"};
            }
        }
    }
    close SYSC;
    
    chomp($ctx->{args}->{processor}->{value});
    chomp($ctx->{args}->{platform}->{value});

    print STDERR indent($ctx)."END: readSystemValues:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return (0, "");
}


sub evaluateCommand
{
    my $ctx = shift;
    my $command   = shift || undef;
    my $mandatory = shift || 0;
    my $cmd       = undef;
    my $out       = undef;
    my @arguments = ();

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: evaluateCommand\n" if($ctx->{time});


    if (!defined $command || $command eq "")
    {
        logPrintError($ctx, "Missing command.\n", 14);
        return undef;
    }

    if ($command =~ /^hwinfo\s*(.*)\s*$/)
    {
        if(!$ctx->{nohwdata})
        {
            $cmd = $ctx->{hwinfo};
            if (defined $1)
            {
                @arguments = split(/\s+/, $1);
            }
        }
        elsif($ctx->{mandatory})
        {
            logPrintError($ctx, "Mandatory hardware data cannot be supplied because the option --no-hw-data was given.\n",
                          3);
            return undef;
        }
        else
        {
            return "DISCARDED";
        }
    }
    elsif ($command =~ /^lsb_release\s*(.*)\s*$/)
    {
        # maybe lsb_release is not installed
        if(-e $ctx->{lsb_release})
        {
            $cmd = $ctx->{lsb_release};
            if (defined $1) 
            {
                @arguments = split(/\s+/, $1);
            }
        }
        else
        {
            return "";
        }
    }
    elsif ($command =~ /^uname\s*(.*)\s*$/)
    {
        $cmd = $ctx->{uname};
        if (defined $1)
        {
            @arguments = split(/\s+/, $1);
        }
    }
    elsif ($command =~ /^zmd-secret$/)
    {
        $cmd = undef;
        open(SEC, "< $ctx->{SECRET_FILE}") or do
        {
            logPrintError($ctx, "Cannot open file $ctx->{SECRET_FILE}: $!\n", 12);
            return undef;
        };
        while(<SEC>)
        {
            $out .= $_;
        }
        close SEC;
    }
    elsif ($command =~ /^zmd-ostarget$/)
    {
        $cmd = undef;

        my ($code, $msg) = rugOSTarget($ctx);
        if($code != 0)
        {
            logPrintError($ctx, $msg, $code);
            return undef;
        }
                
        $out = $ctx->{ostarget};
    }
    elsif ($command =~ /^installed-desktops$/)
    {
        $cmd = undef;

        if(!$ctx->{nohwdata})
        {
            my $kde = 0;
            my $gnome = 0;
            my $x11 = 0;
            
            if($#{$ctx->{installedPatterns}} == -1)
            {
                # ignore errors, getPatterns is not so important
                getPatterns($ctx);
            }
            
            foreach my $pat (@{$ctx->{installedPatterns}})
            {
                if($pat eq "kde")
                {
                    $kde = 1;
                }
                elsif($pat eq "gnome")
                {
                    $gnome = 1;
                }
                elsif($pat eq "x11")
                {
                    $x11 = 1;
                }
            }
            
            $out = "KDE" if($kde && !$gnome);
            $out = "GNOME" if(!$kde && $gnome);
            $out = "KDE+GNOME" if($kde && $gnome);
            $out = "X11" if($x11 && !$kde && !$gnome);
            $out = "NOX11" if(!$x11 && !$kde && !$gnome);
        }
        else
        {
            $out = "DISCARDED";
        }
    }
    else
    {
        $out = "DISCARDED"; # command not allowed; reply DISCARDED
        $cmd = undef;
    }

    if (defined $cmd)
    {
        print "Execute $cmd ".join(" ", @arguments)."\n" if($ctx->{debug});
        
        my $pid = open3(\*IN, \*OUT, \*ERR, $cmd, @arguments) or do 
        {
            logPrintError($ctx, "Cannot execute $cmd ".join(" ", @arguments).": $!\n",13);
            return undef;
        };

        while (<OUT>)
        {
            $out .= "$_";
        }
        #chomp($msg) if(defined $msg && $msg ne "");
        while (<ERR>)
        {
            $out .= "$_";
        }
        close OUT;
        close ERR;
        close IN;
        waitpid $pid, 0;
        chomp($out) if(defined $out && $out ne "");
        if(!defined $out || $out eq "") 
        {
            $out = "";
        }
        print STDERR "LENGTH: ".length($out)."\n" if($ctx->{debug});
    }
    
    print STDERR indent($ctx)."END: evaluateCommand:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;
    
    return $out;
}

sub evalNeedinfo
{
    my $ctx = shift;
    my $tree      = shift || undef;
    my $logic     = shift || "";
    my $indent    = shift || "";
    my $mandatory = shift || 0;
    my $modified  = shift || 0;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: evalNeedinfo\n" if($ctx->{time});

    my $mandstr = "";

    my $nextLogic = $logic;
    if($#{$ctx->{registerReadableText}} >= 0) 
    {
        $indent = $indent."  ";
    }
        
    if (! defined $tree)
    {
        logPrintError($ctx, "Missing data.\n", 14);
        return $modified;
    }
 
    print STDERR "LOGIC: $logic\n" if($ctx->{debug} >= 3);
    print STDERR Data::Dumper->Dump([$tree])."\n" if($ctx->{debug} >= 3);

    foreach my $kid (@$tree)
    {
        my $local_mandatory = $mandatory;
        
        if (ref($kid) eq "SR::param") 
        {
            if (@{$kid->{Kids}} > 1)
            {
                $nextLogic = "AND";
            }
            if($logic eq "") 
            {
                $logic = $nextLogic;
            }
        }
        elsif (ref($kid) eq "SR::select")
        {
            $nextLogic = "OR";
            if($logic eq "")
            {
                $logic = $nextLogic;
            }
        }
        elsif (ref($kid) eq "SR::privacy")
        { 
            if (exists $kid->{description} && defined $kid->{description})
            {
                if(!$ctx->{yastcall})
                {
                    $ctx->{registerPrivPol} .= "\nInformation on Novell's Privacy Policy:\n";
                    $ctx->{registerPrivPol} .= $kid->{description}."\n";
                }
                else
                {
                    $ctx->{registerPrivPol} .= "<p>Information on Novell's Privacy Policy:<br>\n";
                    $ctx->{registerPrivPol} .= $kid->{description}."</p>\n";
                }
            }
            
            if (exists $kid->{url} && defined $kid->{url} && $kid->{url} ne "")
            {
                if(!$ctx->{yastcall})
                {
                    $ctx->{registerPrivPol} .= $kid->{url}."\n";
                }
                else
                {
                    $ctx->{registerPrivPol} .= "<p><a href=\"".$kid->{url}."\">";
                    $ctx->{registerPrivPol} .= $kid->{url}."</a></p>\n";
                }
            }
        }
        elsif (ref($kid) eq "SR::needinfo")
        {
            # do nothing
        }
        else
        {
            # skip host, guid, product and maybe more to come later. 
            # There are no strings for the user to display.
            next;
        }

        if (exists  $kid->{class} &&
            defined $kid->{class} &&
            $kid->{class} eq "mandatory")
        {
            $local_mandatory = 1;
            $mandstr = "(mandatory)";
            print STDERR "Found mandatory\n" if($ctx->{debug} >= 3);
        }
        elsif(!$local_mandatory &&
              !exists $kid->{class})
        {
            $mandstr = "(optional)";
        }
  
        if (ref($kid) ne "SR::privacy" &&
            @{$kid->{Kids}} == 0 &&
            defined $kid->{description} &&
            defined $kid->{id})
        {
            if ( ($ctx->{nooptional} && $local_mandatory) || !$ctx->{nooptional})
            {
                if(! exists $kid->{command})
                {
                    print STDERR "Add instruction\n" if($ctx->{debug} >= 3);
                    
                    my $txt = $indent."* ".$kid->{description}." $mandstr";
                    $ctx->{args}->{$kid->{id}} = { flag => "m", 
                                                   value => undef, 
                                                   kind => ($local_mandatory)?"mandatory":"optional"};
                    
                    if(!$ctx->{yastcall})
                    {
                        $txt .= ":\t".$kid->{id}."=<value>\n";
                    }
                    else
                    {
                        $txt .= "\n";
                    }
                    push @{$ctx->{registerReadableText}}, $txt;
                    $modified  = 1;
                }
                else
                {
                    my $ret = evaluateCommand($ctx, $kid->{command}, $local_mandatory);
                    if($ctx->{errorcode} != 0)
                    {
                        return $modified;
                    }
                    if (defined $ret)
                    {
                        $ctx->{args}->{$kid->{id}} = { flag  => "a", 
                                                       value => $ret,
                                                       kind  => ($local_mandatory)?"mandatory":"optional"
                                                     };
                        $modified = 1;
                    }
                }
            }
        }
        elsif (ref($kid) ne "SR::privacy" && defined $kid->{description})
        {
            if ( ($ctx->{nooptional} && $local_mandatory) || !$ctx->{nooptional})
            {
                print STDERR "Add description\n" if($ctx->{debug} >= 3);
                push @{$ctx->{registerReadableText}}, $indent.$kid->{description}." $mandstr with:\n";
            }
        }

        if ( exists $kid->{Kids} && @{$kid->{Kids}} > 0 )
        {
            $modified = evalNeedinfo($ctx, $kid->{Kids}, $nextLogic, $indent, $local_mandatory, $modified);
            $nextLogic = $logic;
            if (defined $ctx->{registerReadableText}->[$#{$ctx->{registerReadableText}}] &&
                $ctx->{registerReadableText}->[$#{$ctx->{registerReadableText}}] =~ /^\s*AND|OR\s*$/i)
            {
                if ($logic =~ /^\s*$/)
                {
                    pop @{$ctx->{registerReadableText}};
                }
                else
                {
                    $ctx->{registerReadableText}->[$#{$ctx->{registerReadableText}}] = $indent."$logic\n";
                }
            }
            else
            {
                push @{$ctx->{registerReadableText}}, $indent."$logic\n";
            }
        }
    }

    print STDERR indent($ctx)."END: evalNeedinfo:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return $modified;
}


sub walkResultZmdconfig
{
    my $ctx = shift;
    my $tree        = shift || undef;
    my $name        = shift || "zmdConfig";
    my $serviceName = shift || undef;
    my $paramName   = shift || undef;
    my $catalogURL  = shift || undef;
    my $code = 0;
    my $msg = "";

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: walkResultZmdconfig\n" if($ctx->{time});

    if (!defined $tree)
    {
        return logPrintReturn($ctx, "Missing data.\n", 14);
    }

    foreach my $kid (@$tree)
    {
        if (ref($kid) eq "SR::service" && 
            exists $kid->{id})
        {
            $serviceName = $kid->{id};
            if (exists $kid->{type} && defined $kid->{type})
            {
                $ctx->{$name}->{$serviceName}->{type} = $kid->{type};
            }
        }
        
        if ($serviceName &&
            ref($kid) eq "SR::param" )
        {
            if (exists $kid->{id} && defined $kid->{id})
            {
                $paramName = $kid->{id};
            }
            elsif (exists $kid->{name} && defined $kid->{name})
            {                
                $paramName = $kid->{name};
                
                if($paramName eq "catalog") 
                {
                    if(exists $kid->{url} && defined $kid->{url})
                    {
                        $catalogURL = $kid->{url};
                    }
                    else
                    {
                        $catalogURL = undef;
                    }
                }
            }
        }
        
        if ($serviceName && 
            $paramName &&
            ref($kid->{Kids}->[0]) eq "SR::Characters" &&
            exists $kid->{Kids}->[0]->{Text} &&
            defined $kid->{Kids}->[0]->{Text})
        {
            my $text = $kid->{Kids}->[0]->{Text};
            
            if ($paramName eq "catalog") # more than one catalog allowed
            {                
                if(defined $catalogURL && $catalogURL ne "")
                {
                    $ctx->{$name}->{$serviceName}->{catalog}->{$text}->{url} = $catalogURL;
                    $ctx->{$name}->{$serviceName}->{alias} = $serviceName;
                }
                else
                {
                    $ctx->{$name}->{$serviceName}->{alias} = $text;
                }
            }
            else
            {
                if(lc($paramName) eq "url" && $text !~ /\/$/)
                {
                    $ctx->{$name}->{$serviceName}->{$paramName} = $text."/";
                }
                else
                {
                    $ctx->{$name}->{$serviceName}->{$paramName} = $text;
                }
            }
        }
        
        if (!defined $serviceName &&
            ref($kid) eq "SR::param")
        {
            if (exists $kid->{id})
            {                
                $paramName = $kid->{id};
            }
        }
        
        if (! defined $serviceName && 
            $paramName &&
            ref($kid->{Kids}->[0]) eq "SR::Characters" &&
            exists $kid->{Kids}->[0]->{Text} &&
            defined $kid->{Kids}->[0]->{Text})
        {
            $ctx->{$name}->{"globalzmdoptions"}->{$paramName} = $kid->{Kids}->[0]->{Text};
        }
        
        if (exists $kid->{Kids} &&
            ref($kid->{Kids}) eq "ARRAY" &&
            ref($kid) ne "SR::Characters")   #check on Kids does not work in Characters
        {
            ($code, $msg) = walkResultZmdconfig($ctx, $kid->{Kids}, $name, $serviceName, $paramName, $catalogURL);
            if($code != 0)
            {
                return ($code, $msg);
            }
            if (defined $paramName)
            {
                $paramName = undef;
            }
            elsif (defined $serviceName)
            {
                $serviceName = undef;
            }
        }
    }

    print STDERR indent($ctx)."END: walkResultZmdconfig:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return (0, "");
}

sub buildXML
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: buildXML\n" if($ctx->{time});
    
    my $output = '<?xml version="1.0" encoding="utf-8"?>';
    
    my $writer = new XML::Writer(OUTPUT => \$output);

    my %a = ("xmlns" => "http://www.novell.com/xml/center/regsvc-1_0",
             "client_version" => "$SRversion");
    
    if(!$ctx->{nooptional})
    {
        $a{accept} = "optional";
    }
    if($ctx->{acceptmand} || $ctx->{nooptional}) 
    {
        $a{accept} = "mandatory";
    }
    if($ctx->{forcereg}) 
    {
        $a{force} = "registration";
    }
    if($ctx->{batch}) 
    {
        $a{force} = "batch";
    }
    
    $writer->startTag("register", %a);
    
    $writer->startTag("guid");
    $writer->characters($ctx->{guid});
    $writer->endTag("guid");

    if(defined $ctx->{hostGUID} && $ctx->{hostGUID} ne "") 
    {
        $writer->startTag("host");
        $writer->characters($ctx->{hostGUID});
        $writer->endTag("host");
    }
    else
    {
        $writer->emptyTag("host");
    }
    
    foreach my $PArray (@{$ctx->{products}})
    {
        if(defined $PArray->[0] && $PArray->[0] ne "" &&
           defined $PArray->[1] && $PArray->[1] ne "")
        {
            $writer->startTag("product",
                              "version" => $PArray->[1],
                              "release" => $PArray->[2],
                              "arch"    => $PArray->[3]);
            if ($PArray->[0] =~ /\s+/)
            {
                $writer->cdata($PArray->[0]);
            }
            else
            {
                $writer->characters($PArray->[0]);
            }
            $writer->endTag("product");
        }
    }
    
    foreach my $key (keys %{$ctx->{args}})
    {
        next if(!defined $ctx->{args}->{$key}->{value});

        if($ctx->{args}->{$key}->{value} eq "")
        {
            $writer->emptyTag("param", "id" => $key);
        }
        else
        {
            $writer->startTag("param",
                              "id" => $key);
            if ($ctx->{args}->{$key}->{value} =~ /\s+/)
            {
                $writer->cdata($ctx->{args}->{$key}->{value});
            }
            else
            {
                $writer->characters($ctx->{args}->{$key}->{value});
            }
            $writer->endTag("param");
        }
    }

    # request up-to 5 mirrors in <zmdconfig> to 
    $writer->emptyTag("mirrors", "count" => $ctx->{mirrorCount});

    $writer->endTag("register");

    print STDERR "XML:\n$output\n" if($ctx->{debug} >= 3);

    print STDERR indent($ctx)."END: buildXML:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return $output;
}


sub sendData
{
    my $ctx = shift;
    my $url  = shift || undef;
    my $data = shift || undef;
    
    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: sendData\n" if($ctx->{time});
    
    my $curlErr = 0;
    my $res = "";
    my $err = "";
    my %header = ();
    my $code = "";
    my $mess = "";
    
    if (! defined $url)
    {
        logPrintError($ctx, "Cannot send data to registration server. Missing URL.\n", 14);
        return;
    }
    if($url =~ /^-/)
    {
        logPrintError($ctx, "Invalid protocol($url).\n", 15);
        return;
    }

    my $uri = URI->new($url);
    
    if(!defined $uri->host || $uri->host !~ /$ctx->{initialDomain}$/)
    {
        logPrintError($ctx, "Invalid URL($url). Data could only be send to $ctx->{initialDomain} .\n", 15);
        return;
    }
    if(!defined $uri->scheme || $uri->scheme ne "https")
    {
        logPrintError($ctx, "Invalid protocol($url). https is required.\n", 15);
        return;
    }
    $url = $uri->as_string;
        
    if (! defined $data)
    {
        logPrintError($ctx, "Cannot send data. Missing data.\n", 14);
        return;
    }

    my @cmdArgs = ( "--capath", $ctx->{CA_PATH});

    my $fh = new File::Temp(TEMPLATE => 'dataXXXXX',
                            SUFFIX   => '.xml',
                            DIR      => '/tmp/');
    print $fh $data;

    push @cmdArgs, "--data", "@".$fh->filename();
    push @cmdArgs, "-i";
    push @cmdArgs, "--max-time", "60";

    foreach my $extraOpt (@{$ctx->{extraCurlOption}})
    {
        if($extraOpt =~ /^([\w-]+)[\s=]*(.*)/)
        {
            if(defined $1 && $1 ne "")
            {
                push @cmdArgs, $1;
                
                if(defined $2 && $2 ne "")
                {
                    push @cmdArgs, $2;
                }
            }
        }
    }
    
    push @cmdArgs, "$url";

    print STDERR "Call $ctx->{curl} ".join(" ", @cmdArgs)."\n" if($ctx->{debug} >= 2);
    print STDERR "SEND DATA to URI: $url:\n" if($ctx->{debug} >= 2);
    print STDERR "$data\n"  if($ctx->{debug} >= 2);

    print {$ctx->{LOGDESCR}} "\nSEND DATA to URI: $url:\n" if(defined $ctx->{LOGDESCR});
    print {$ctx->{LOGDESCR}} "$data\n"  if(defined $ctx->{LOGDESCR});

    if($ctx->{noproxy})
    {
        delete $ENV{'http_proxy'};
        delete $ENV{'HTTP_PROXY'};
        delete $ENV{'https_proxy'};
        delete $ENV{'HTTPS_PROXY'};
        delete $ENV{'ALL_PROXY'};
        delete $ENV{'all_proxy'};
    }
    $ENV{'PATH'} = '/bin:/usr/bin:/sbin:/usr/sbin:/opt/kde3/bin/:/opt/gnome/bin/';

    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{curl}, @cmdArgs) or do {
        logPrintError($ctx, "Cannot execute $ctx->{curl} ".join(" ", @cmdArgs).": $!\n",13);
        return;
    };

    my $foundBody = 0;
    while (<OUT>)
    {
        $res = "" if(! defined $res);
        if ($foundBody)
        {
            $res .= "$_";
        }
        elsif ($_ =~ /^HTTP\/\d\.\d\s(\d+)\s(.*)$/)
        {
            if (defined $1 && $1 ne "")
            {
                $code = $1;
            }
            if (defined $2 && $2 ne "")
            {
                $mess = $2;
            }
        }
        elsif ($_ =~ /^[\w-]+:\s/)
        {
            my ($key, $val) = split(": ", $_, 2);
            $header{$key} = $val;
        }
        elsif ($_ =~ /^\s*</)
        {
            $foundBody = 1;
            $res .= "$_";
        }
    }
    while (<ERR>)
    {
        $err .= "$_";
    }
    close OUT;
    close ERR;
    close IN;
    waitpid $pid, 0;

    $curlErr = ($?>>8);

    print STDERR "CURL RETURN WITH: $curlErr\n" if($ctx->{debug} >= 2);
    print STDERR "\nRECEIVED DATA:\n" if($ctx->{debug} >= 2);
    print STDERR "CODE: $code MESSAGE:  $mess\n" if($ctx->{debug} >= 2);
    print STDERR "HEADER: ".Data::Dumper->Dump([\%header])."\n" if($ctx->{debug} >= 2);
    print STDERR "BODY:  $res\n" if($ctx->{debug} >= 2);
    
    print {$ctx->{LOGDESCR}} "\nRECEIVED DATA:\n" if(defined $ctx->{LOGDESCR});
    print {$ctx->{LOGDESCR}} "CURL RETURN WITH: $curlErr\n" if(defined $ctx->{LOGDESCR});
    print {$ctx->{LOGDESCR}} "CODE: $code MESSAGE:  $mess\n" if(defined $ctx->{LOGDESCR});
    print {$ctx->{LOGDESCR}} "HEADER: ".Data::Dumper->Dump([\%header])."\n" if(defined $ctx->{LOGDESCR});
    print {$ctx->{LOGDESCR}} "BODY:  $res\n" if(defined $ctx->{LOGDESCR});

    if ($curlErr != 0)
    {
        logPrintError($ctx, "Execute curl command failed with '$curlErr': $err", 4);
        return $res;
    }

    if ($code >= 300 && exists $header{Location} && defined $header{Location})
    {
        if ($ctx->{redirects} > 5)
        {
            logPrintError($ctx, "Too many redirects. Aborting.\n", 5);
            return $res;
        }
        $ctx->{redirects}++;
        
        my $loc = $header{Location};

        local $/ = "\r\n";
        chomp($loc);
        local $/ = "\n";

        #print STDERR "sendData(redirect): ".(tv_interval($t0))."\n" if($ctx->{time});

        $res = sendData($ctx, $loc, $data);
    }
    elsif($code < 200 || $code >= 300) 
    {
        my $b = "";
        my @c = ();

        if(-e "/usr/bin/lynx")
        {
            $b = "/usr/bin/lynx";
            push @c, "-dump", "-stdin";
        }
        elsif(-e "/usr/bin/w3m") 
        {
            $b = "/usr/bin/w3m";
            push @c, "-dump", "-T", "text/html";
        }
        
        my $out = "";

        my $pid = open3(\*IN, \*OUT, \*ERR, $b, @c) or do
        {
            logPrintError($ctx, "Cannot execute $b ".join(" ", @c).": $!\n",13);
            return undef;
        };
        
        print IN $res;
        close IN;
        
        while (<OUT>)
        {
            $out .= "$_";
        }
        #chomp($msg) if(defined $msg && $msg ne "");
        while (<ERR>)
        {
            $out .= "$_";
        }
        close OUT;
        close ERR;
        waitpid $pid, 0;
        chomp($out) if(defined $out && $out ne "");
        
        $out .= "\n$mess\n";
        
        logPrintError($ctx, "ERROR: $code: $out\n", 2);
    }
    #else
    #{
        print STDERR indent($ctx)."END: sendData:".(tv_interval($t0))."\n" if($ctx->{time});
        $ctx->{timeindent}--;
    #}
    
    return $res;
}


sub getPatterns
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: getPatterns\n" if($ctx->{time});

    my $code = 0;
    my $msg = "";
    
    if(!defined $ctx->{querypool})
    {
        # SLES9 ?
        return (0, "");
    }
    

    print STDERR "query pool command: $ctx->{querypool} patterns \@system \n" if($ctx->{debug} >= 1);
    
    my $result = `$ctx->{querypool} patterns \@system`;
    $code = ($?>>8);

    if($code != 0) 
    {
        $code += 30;
        $msg = "Query patterns failed. $result\n";
        #syslog("err", "Query patterns failed($code): $result");
    }
    else 
    {
        foreach my $line (split("\n", $result))
        {
            next if($line =~ /^\s*$/);
            
            my @p = split('\|', $line);
            
            if(defined $p[0] && $p[0] eq "i" && 
               defined $p[2] && $p[2] ne "")
            {
                push @{$ctx->{installedPatterns}}, $p[2];
            }
        }
    }
    
    print STDERR "Query patterns failed($code): $result\n" if($ctx->{debug} && $code != 0);
    
    print STDERR "installed patterns:           ".Data::Dumper->Dump([$ctx->{installedPatterns}])."\n" if($ctx->{debug});
    
    print STDERR indent($ctx)."END: getPatterns:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return logPrintReturn($ctx, $msg, $code);
}

sub getProducts
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: getProducts\n" if($ctx->{time});

    my $code = 0;
    my $msg = "";
    my $result = "";
    
    if(defined $ctx->{querypool})
    {
        print STDERR "query pool command: $ctx->{querypool} products \@system \n" if($ctx->{debug} >= 1);
        
        $result = `$ctx->{querypool} products \@system`;
        $code = ($?>>8);
        
        if($code != 0) 
        {
            $code += 30;
            $msg = "Query products failed. $result\n";
            #syslog("err", "Query products failed($code): $result");
        }
        else 
        {
            foreach my $line (split("\n", $result))
            {
                next if($line =~ /^\s*$/);
                
                my @p = split('\|', $line);
                my $installed   = $p[0];
                my $type        = $p[1];
                my $product     = $p[2];
                my $version     = $p[3];
                my $arch        = $p[4];
                my $distproduct = "";
                my $distversion = "";
                my $distrelease = "";
                
                if(defined $p[5] && $p[5] ne "")
                {
                    $distproduct = $p[5];
                }
                else
                {
                    $distproduct = $product;
                }
                if(defined $p[6] && $p[6] ne "")
                {
                    $distversion = $p[6];
                }
                else
                {
                    $distversion = $version;
                }
                
                if(!defined $arch || $arch eq "" || $arch eq "noarch")
                {
                    $arch = `$ctx->{uname} -i`;
                    chomp($arch);
                }
                
                if(defined $distversion && $distversion ne "")
                {
                    my @v = split("-", $distversion, 2);
                    if(exists $v[0] && defined $v[0] && $v[0] ne "")
                    {
                        $distversion = $v[0];
                    }
                    if(exists $v[1] && defined $v[1] && $v[1] ne "")
                    {
                        $distrelease = $v[1];
                    }
                }
                
                if($installed eq "i" && lc($type) eq lc("product"))
                {
                    push @{$ctx->{installedProducts}}, [$distproduct, $distversion, $distrelease, $arch];
                }
            }
        }
    }
    elsif( -e $ctx->{suserelease})
    {
        # SLES9

        my $product = `$ctx->{suserelease} --distproduct -s`;
        $code = ($?>>8);
        if($code != 0) 
        {
            $code += 40;
            $msg = "Query products failed. \n";
            #syslog("err", "Query products failed($code): $result");
        }

        my $version = `$ctx->{suserelease} --distversion -s`;
        $code = ($?>>8);
        if($code != 0) 
        {
            $code += 40;
            $msg = "Query products failed. \n";
            #syslog("err", "Query products failed($code): $result");
        }

        my $arch = `$ctx->{uname} -m`;
        $code = ($?>>8);
        if($code != 0) 
        {
            $code += 40;
            $msg = "Query products failed. \n";
            #syslog("err", "Query products failed($code): $result");
        }
        chomp($product);
        chomp($version);
        chomp($arch);       
        
        push @{$ctx->{installedProducts}}, [$product, $version, "", $arch];
    }
    
    print STDERR "Query products failed($code): $result\n" if($ctx->{debug} && $code != 0);
    
    print STDERR "installed products:           ".Data::Dumper->Dump([$ctx->{installedProducts}])."\n" if($ctx->{debug});
    syslog("info", "Installed Products Dump: ".Data::Dumper->Dump([$ctx->{installedProducts}]));


    print STDERR indent($ctx)."END: getProducts:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return logPrintReturn($ctx, $msg, $code);
}

sub getZmdConfigValues
{
    my $ctx = shift;
    my $tree = shift || undef;
    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: getZmdConfigValues\n" if($ctx->{time});
    
    if (!defined $tree) 
    {
        return logPrintReturn($ctx, "Missing data.\n", 14);
    }
    
    #print Data::Dumper->Dump([$tree]);
    
    my ($code, $msg) = zmdProxyConfig($ctx);
    if($code != 0)
    {
        return ($code, $msg);
    }
    
    if($ctx->{addRegSrvSrc})
    {
        
        ($code, $msg) = walkResultZmdconfig($ctx, $tree, "tmpZmdConfig");
        if($code != 0)
        {
            delete $ctx->{tmpZmdConfig};
            return ($code, $msg);
        }

        foreach my $oldService (keys %{$ctx->{tmpZmdConfig}})
        {
            # do zmd configuration directly here
            
            # ignore possible errors
            rugPreferences($ctx, "hardware-inventory-enabled", "false");
            rugPreferences($ctx, "software-inventory-enabled", "false");
            
            if($oldService eq "globalzmdoptions")
            {
                if(exists $ctx->{tmpZmdConfig}->{globalzmdoptions}->{update_inventory} &&
                   defined $ctx->{tmpZmdConfig}->{globalzmdoptions}->{update_inventory} &&
                   !$ctx->{nohwdata} && !$ctx->{nooptional})
                {
                    # ignore possible errors
                    rugPreferences($ctx, "hardware-inventory-enabled",
                                   $ctx->{tmpZmdConfig}->{globalzmdoptions}->{update_inventory});
                    rugPreferences($ctx, "software-inventory-enabled",
                                   $ctx->{tmpZmdConfig}->{globalzmdoptions}->{update_inventory});
                }
                next;
            }
            my $url = $ctx->{tmpZmdConfig}->{$oldService}->{url};
            $ctx->{zmdConfig}->{$url} = {};
            copyService($ctx->{zmdConfig}->{$url}, $ctx->{tmpZmdConfig}->{$oldService});
        }
        delete $ctx->{tmpZmdConfig};
    }
    
    foreach my $src (@{$ctx->{addAdSrc}})
    {
        my $uri = URI->new($src);
        my $alias = $uri->query_param("alias");
        
        $ctx->{zmdConfig}->{$src}->{'url'} = $src;
        $ctx->{zmdConfig}->{$src}->{'type'} = "zypp";
        if(defined $alias && $alias ne "")
        {
            $ctx->{zmdConfig}->{$src}->{'alias'} = $alias;
        }
    }
    
    print Data::Dumper->Dump([$ctx->{zmdConfig}]) if($ctx->{debug} >= 2);

    # HACK: write urls for yast
    if(defined $ctx->{dumpfilehack} && $ctx->{dumpfilehack} ne "")
    {
        my $exc = sysopen(XMLD, $ctx->{dumpfilehack}, O_CREAT|O_TRUNC|O_WRONLY, 0600) and do
        {
            foreach my $service (keys %{$ctx->{zmdConfig}})
            {
                next if($service eq "globalzmdoptions");
                
                if($ctx->{zmdConfig}->{$service}->{type} eq "yum" ||
                   $ctx->{zmdConfig}->{$service}->{type} eq "zypp")
                {
                    my $name = $service;
                    if(exists $ctx->{zmdConfig}->{$service}->{alias} &&
                       defined $ctx->{zmdConfig}->{$service}->{alias})
                    {
                        $name = $ctx->{zmdConfig}->{$service}->{alias};
                    }
                    elsif(exists $ctx->{zmdConfig}->{$service}->{catalog} &&
                          ref($ctx->{zmdConfig}->{$service}->{catalog}) eq "HASH")
                    {
                        my @x = keys %{$ctx->{zmdConfig}->{$service}->{catalog}};
                        
                        if(exists $x[0] && defined $x[0] && $x[0] ne "")
                        {
                            $name = $x[0];
                        }
                    }
                    
                    my $curi = URI->new($ctx->{zmdConfig}->{$service}->{url});
                    $curi->query_form(alias => $name);

                    print XMLD $curi->as_string."\n";
                    syslog("info", "dumpfile write(zypp/yum):".$curi->as_string);
                }
                elsif($ctx->{zmdConfig}->{$service}->{type} eq "nu" ||
                      $ctx->{zmdConfig}->{$service}->{type} eq "rce")
                {
                    foreach my $u (keys %{$ctx->{zmdConfig}->{$service}->{catalog}})
                    {
                        my $url = fillURL($ctx, $ctx->{zmdConfig}->{$service}->{catalog}->{$u}->{url},
                                          {alias => $u});
                        print stripURL($ctx, $url)."\n" if($ctx->{debug});
                        
                        print XMLD "$url\n";
                        syslog("info", "dumpfile write(nu):".$u);
                    }
                }
            }
            close XMLD;
        };

        if(!defined $exc || $exc == 0)
        {
            syslog("err", "Cannot write to $ctx->{dumpfilehack}: $!");
        }
    }

    print STDERR indent($ctx)."END: getZmdConfigValues:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;
    
    return (0, "");
}

sub saveLastZmdConfig
{
    my $ctx = shift;
    #my $zmdconfig = shift || undef;
    
    if(!exists $ctx->{newzmdconfig} || !defined $ctx->{newzmdconfig})
    {
        return logPrintReturn($ctx, "Missing data.\n", 14);
    }
    
    if(!-d "/var/cache/SuseRegister/")
    {
        mkdir "/var/cache/SuseRegister/";
    }
    
    # make a backup of the old cache file
    if(-e $ctx->{zmdcache})
    {
        copy($ctx->{zmdcache}, $ctx->{zmdcache}.".old");
    }
    
    open(OUT, "> ".$ctx->{zmdcache}) or return logPrintReturn($ctx, "Cannot open cache file '$ctx->{zmdcache}': $!", 12);
    print OUT $ctx->{newzmdconfig};
    close OUT;

    return (0, "");
}

sub readLastZmdConfig
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: readLastZmdConfig\n" if($ctx->{time});

    $ctx->{"lastZmdConfig"} = undef;

    if(-e $ctx->{zmdcache})
    {
        my $p = new XML::Parser(Style => 'Objects', Pkg => 'SR');
        my $tree = $p->parsefile($ctx->{zmdcache});
        
        my ($code, $msg) = walkResultZmdconfig($ctx, $tree->[0]->{Kids}, "tmpLastZmdConfig");
        if($code != 0)
        {
            delete $ctx->{tmpLastZmdConfig};
            return ($code, $msg);
        }
        
        foreach my $oldService (keys %{$ctx->{tmpLastZmdConfig}})
        {
            my $url = $ctx->{tmpLastZmdConfig}->{$oldService}->{url};
            $ctx->{lastZmdConfig}->{$url} = {};
            copyService($ctx->{lastZmdConfig}->{$url}, $ctx->{tmpLastZmdConfig}->{$oldService});
        }
        delete $ctx->{tmpLastZmdConfig};
    }
    
    print STDERR indent($ctx)."END: readLastZmdConfig:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return (0, "");
}


sub logPrintReturn
{
    my $ctx = shift;
    my $message = shift || "";
    my $code    = shift || 0;

    if($code != 0)
    {
        syslog("err", "$message($code)");
        print STDERR "$message($code)\n" if($ctx->{debug});
    }
    
    # cleanup errors in the context
    $ctx->{errorcode} = 0;
    $ctx->{errormsg} = "";

    return ($code, $message);
}


sub logPrintError
{
    my $ctx = shift;
    my $message = shift || undef;
    my $code    = shift || 0;

    if($code != 0) 
    {
        
        if(exists $ctx->{args}->{password})
        {
            $ctx->{args}->{password}->{value} = "secret";
        }
        if(exists $ctx->{args}->{passwd})
        {
            $ctx->{args}->{passwd}->{value} = "secret";
        }
        if(exists $ctx->{args}->{secret})
        {
            $ctx->{args}->{secret}->{value} = "secret";
        }
        my $cmdtxt = "Commandline params: no-optional:$ctx->{nooptional}  forceregistration:$ctx->{forcereg}  ";
        $cmdtxt .= "no-hw-data:$ctx->{nohwdata} batch:$ctx->{batch} ";
        
        syslog("err", $cmdtxt);
        syslog("err", "Argument Dump: ".Data::Dumper->Dump([$ctx->{args}]));
        syslog("err", "Products Dump: ".Data::Dumper->Dump([$ctx->{products}]));
        syslog("err", "$message($code)");
    }
    
    $ctx->{errorcode} = $code;
    $ctx->{errormsg} = $message;
    
    return;
}

sub listProducts
{
    my $ctx = shift;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: listProducts\n" if($ctx->{time});
    
    my $output = "\n";
    
    my $writer = new XML::Writer(OUTPUT => \$output);

    $ctx->{redirects} = 0;

    my $res = sendData($ctx, $ctx->{URL}."?".$ctx->{URLlistProducts}."&lang=en-US&version=$ctx->{version}", $output);
    if($ctx->{errorcode} != 0)
    {
        return logPrintReturn($ctx, $ctx->{errormsg}, $ctx->{errorcode});
    }
    
    my $p = new XML::Parser(Style => 'Objects', Pkg => 'SR');
    my $tree = $p->parse($res);
    
    #print Data::Dumper->Dump([$tree])."\n";
    
    if (! defined $tree || ref($tree->[0]) ne "SR::productlist")
    {
        return logPrintReturn($ctx, "Unknown XML format. Cannot show human readable output. Try --xml-output.\n",
                              6);
    }
    
    foreach my $kid (@{$tree->[0]->{Kids}})
    {
        #print Data::Dumper->Dump([$kid])."\n";
        
        if (ref($kid) eq "SR::product" &&
            exists  $kid->{Kids} &&
            exists  $kid->{Kids}->[0] &&
            ref($kid->{Kids}->[0]) eq "SR::Characters" &&
            exists  $kid->{Kids}->[0]->{Text} &&
            defined $kid->{Kids}->[0]->{Text} &&
            $kid->{Kids}->[0]->{Text} ne "")
        {
                #print "SEE:".Data::Dumper->Dump([$tree->[1]->[$i]])."\n\n";
            
            push @{$ctx->{serverKnownProducts}}, [$kid->{Kids}->[0]->{Text}, "0"];
        }
    }

    print STDERR "Server Known Products:".Data::Dumper->Dump([$ctx->{serverKnownProducts}])."\n" if($ctx->{debug} >= 2);

    print STDERR indent($ctx)."END: listProducts:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return (0, "");
}

sub intersection
{
    my $ctx = shift;
    my $arr1 = shift || undef;
    my $arr2 = shift || undef;
    my $ret = [];
    
    if(!defined $arr1 || !defined $arr2 || 
       ref($arr1->[0]) ne "ARRAY" || ref($arr2->[0]) ne "ARRAY")
    {
        return [];
    }

    print STDERR "intersect1: ".Data::Dumper->Dump([$arr1])."\n" if($ctx->{debug} >= 3);
    print STDERR "intersect2: ".Data::Dumper->Dump([$arr2])."\n" if($ctx->{debug} >= 3);
    
    foreach my $v1 (@$arr1)
    {
        foreach my $v2 (@$arr2) 
        {
            if(lc($v1->[0]) eq lc($v2->[0]))
            {
                if($v2->[1] ne "0")
                {
                    push @$ret, $v2;
                }
                else
                {
                    push @$ret, $v1;
                }
                last;
            }
        }
    }
    
    print STDERR "intersect return : ".Data::Dumper->Dump([$ret])."\n" if($ctx->{debug} >= 3);
    return $ret;
}


sub currentServices
{
    my $ctx = shift;
    my $code = 1;
    my $msg = "";
    my $msg2 = "";
    my $catalogs = {};
    
    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: currentServices\n" if($ctx->{time});

    if($ctx->{rugzmdInstalled})
    {
        $msg = `$ctx->{rug} --terse --no-abbrev sl`;
        $code = ($?>>8);

        if ($msg =~ /^---\sNo services found\s---/)
        {
            return (0, "");
        }
    
        if($code != 0)
        {
            $code += 20;
            return logPrintReturn($ctx, "Cannot list services\n", $code);
        }
    

        $msg2 = `$ctx->{rug} --terse --no-abbrev ca`;
        $code = ($?>>8);
        if($code != 0)
        {
            $code += 20;
            return logPrintReturn($ctx, "Cannot list catalogs\n", $code);
        }
    }
    elsif($ctx->{zypperInstalled})
    {
        $msg = `$ctx->{zypper} --rug-compatible sl`;
        $code = ($?>>8);

        if($code != 0)
        {
            $code += 20;
            return logPrintReturn($ctx, "Cannot list services\n", $code);
        }
    }
    
    foreach my $line2 (split("\n", $msg2))
    {
        my($sub, $catName, $service) = split('\s*\|\s*', $line2, 3);
        if(defined $sub && defined $catName && defined $service)
        {
            print STDERR "Found $sub, $catName, $service\n" if($ctx->{debug} >= 2);
        
            $catalogs->{$service}->{$catName} = (lc($sub) eq "yes")?1:0 ;
        }
    }
    

    foreach my $line (split("\n", $msg))
    {
        my($num, $stat, $type, $name, $uri) = split('\s*\|\s*', $line, 5);
        if(defined $num && defined $stat && defined $type && defined $name && defined $uri)
        {
            # try to drop the headline
            next if($num eq "#");
            # the line --+------+------+... will not appear here
    
            my $strURL = stripURL($ctx, $uri);
            if($strURL eq "")
            {
                return logPrintReturn($ctx, "Missing URL\n", 14);
            }
            $uri = $strURL;
            
            print STDERR "Found $num, $stat, $type, $name, $uri\n" if($ctx->{debug} >= 2);

            if(!$ctx->{rugzmdInstalled})
            {
                # we used zypper to get these values. All types are zypp
                $type = "zypp";
            }
            
            $ctx->{tmpCurrentSources}->{$name}->{type} = $type;
            $ctx->{tmpCurrentSources}->{$name}->{url} = $uri;
            if(lc($type) eq "zypp" || lc($type) eq "yum")
            {
                $ctx->{tmpCurrentSources}->{$name}->{alias} = $name;
            }
            else
            {
                foreach my $n (keys %{$catalogs->{$name}})
                {
                    $ctx->{tmpCurrentSources}->{$name}->{catalog}->{$n}->{subscribed} = $catalogs->{$name}->{$n};
                }
            }
        }
    }
    
    foreach my $oldService (keys %{$ctx->{tmpCurrentSources}})
    {
        my $url = $ctx->{tmpCurrentSources}->{$oldService}->{url};
        $ctx->{currentSources}->{$url} = {};
        copyService($ctx->{currentSources}->{$url}, $ctx->{tmpCurrentSources}->{$oldService});
    }
    delete $ctx->{tmpCurrentSources};

    print STDERR indent($ctx)."END: currentServices:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;
    
    return (0, "");
}

sub copyService
{
    my $dst = shift || undef;
    my $src = shift || undef;

    $dst->{type}    = $src->{type}    if(exists $src->{type});
    $dst->{url}     = $src->{url}     if(exists $src->{url});
    $dst->{regcode} = $src->{regcode} if(exists $src->{regcode});
    $dst->{alias}   = $src->{alias}   if(exists $src->{alias});

    if(exists $src->{catalog} && ref($src->{catalog}) eq "HASH")
    {
        foreach my $c (keys %{$src->{catalog}})
        {
            if(ref($src->{catalog}->{$c}) eq "HASH")
            {
                foreach my $hw (keys %{$src->{catalog}->{$c}})
                {
                    $dst->{catalog}->{$c}->{$hw} = $src->{catalog}->{$c}->{$hw};
                }
            }
            else
            {
                $dst->{catalog}->{$c} = $src->{catalog}->{$c};
            }
        }
    }
}

sub zmdProxyConfig
{
    my $ctx = shift;
    my $proxyUrl = "";
    my $code = 0;
    my $msg = "";
    
    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: zmdProxyConfig\n" if($ctx->{time});

    if(exists $ENV{'ALL_PROXY'}   && defined $ENV{'ALL_PROXY'}   && $ENV{'ALL_PROXY'} =~ /^http/ )
    {
        $proxyUrl = $ENV{'ALL_PROXY'};
    }
    
    if(exists $ENV{'HTTPS_PROXY'} && defined $ENV{'HTTPS_PROXY'} && $ENV{'HTTPS_PROXY'} =~ /^http/ )
    {
        $proxyUrl = $ENV{'HTTPS_PROXY'};
    }
    
    if(exists $ENV{'https_proxy'} && defined $ENV{'https_proxy'} && $ENV{'https_proxy'} =~ /^http/ )
    {
        $proxyUrl = $ENV{'https_proxy'};
    }

    if($ctx->{noproxy} || $proxyUrl eq "")
    {
        ($code, $msg) = rugPreferences($ctx, "proxy-url"     , "");
        if($code != 0)
        {
            return ($code, $msg);
        }
        ($code, $msg) = rugPreferences($ctx, "proxy-username", "");
        if($code != 0)
        {
            return ($code, $msg);
        }
        ($code, $msg) = rugPreferences($ctx, "proxy-password", "");
        if($code != 0)
        {
            return ($code, $msg);
        }
    }
    elsif($proxyUrl ne "")
    {
        my $puser = "";
        my $ppass = "";
    
        if(-e "$ENV{HOME}/.curlrc") 
        {
            open(IN, "< $ENV{HOME}/.curlrc") or return logPrintReturn($ctx, "Cannot open .curlrc: $!\n", 13);
            while(<IN>)
            {
                if( $_ =~ /^\s*proxy-user\s*=\s*"?([^:]+):([^":]+)"?\s*$/)
                {
                    $puser = $1;
                    $ppass = $2;
                    last;
                }
            }
            close IN;
        }
        
        ($code, $msg) = rugPreferences($ctx, "proxy-url"     , $proxyUrl);
        if($code != 0)
        {
            return ($code, $msg);
        }
        ($code, $msg) = rugPreferences($ctx, "proxy-username", $puser);
        if($code != 0)
        {
            return ($code, $msg);
        }
        ($code, $msg) = rugPreferences($ctx, "proxy-password", $ppass);
        if($code != 0)
        {
            return ($code, $msg);
        }
    }

    print STDERR indent($ctx)."END: zmdProxyConfig:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return (0,"");
}

sub rugPreferences
{
    my $ctx = shift;
    my $pref  = shift || undef;
    my $value = shift || "";
    my $msg   = "";
    my $code  = 0;
    
    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: rugPreferences\n" if($ctx->{time});

    if(!$ctx->{rugzmdInstalled})
    {
        # If rug/zmd are not installed we are done here.
        syslog("info", "rug/zmd are not installed. finish.");
        return (0, "");
    }

    if (!defined $pref || $pref eq "") 
    {
        return logPrintReturn($ctx, "Missing preference value\n", 14);
    }
    
    # empty value could be a valid value
    if (!defined $value )
    {
        $value = "";
    }
    
    my @rugArgs = ("--quiet", "set", "$pref", "$value");
            
    print STDERR "rug set-pref command: $ctx->{rug} ".join(" ",@rugArgs)."\n" if($ctx->{debug} >= 1);

    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{rug}, @rugArgs) or do {
        return logPrintReturn($ctx, "Cannot execute $ctx->{rug} ".join(" ", @rugArgs).": $!\n", 13);
    };
    
    while (<OUT>)
    {
        $msg .= "$_";
    }
    while (<ERR>)
    {
        $msg .= "$_";
    }
    close IN;
    close OUT;
    close ERR;
    waitpid $pid, 0;
    
    $code = ($?>>8);
    if($code != 0)
    {
        $code += 20;
    }

    $msg =~ s/^\s*//;
    $msg =~ s/\s*$//;
    
    print STDERR "Error during set preferences($code): $msg\n" if($ctx->{debug} && $code != 0);
    
    print STDERR indent($ctx)."END: rugPreferences:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return ($code,$msg);
}


sub zyppServiceAdd
{
    my $ctx = shift;
    my $url     = shift || undef;
    my $type    = shift || undef;
    my $id      = shift || undef;
    my $regcode = shift || undef;
    
    my $msg = "";
    my $code = 0;
    
    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: zyppServiceAdd\n" if($ctx->{time});

    if($ctx->{nozypp} || !$ctx->{zypperInstalled} || $ctx->{nozypper})
    {
        return (0,"");
    }
    
    if (! defined $url || $url eq "") 
    {
        return logPrintReturn($ctx, "Missing URL.\n", 14);
    }
    if (! defined $type || (lc($type) ne "yum" && lc($type) ne "zypp")) 
    {
        return logPrintReturn($ctx, "Invalid type", 14);
    }

    if($ctx->{nozypp})
    {
        return (0,"");
    }
    
    # do not ask questions, especialy not to accept keys.
    # The source is only added to the repo. The questions
    # were asked at the first time this source is used.
    my @rugArgs = ("--non-interactive", "--no-gpg-checks", "sa");
    
    push @rugArgs, "$url";
    
    if(defined $id)
    {
        push @rugArgs, $id;
    }
    else
    {
        # zypper requires a alias name. Use the url as fallback
        push @rugArgs, $url;
    }
    
    print STDERR "zypper add service command: $ctx->{zypper} ".join(" ",@rugArgs)."\n" if($ctx->{debug} >= 1);
    
    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{zypper}, @rugArgs) or do {
        return logPrintReturn($ctx, "Cannot execute $ctx->{zypper} ".join(" ", @rugArgs).": $!\n",13);
    };
    
    while (<OUT>)
    {
        $msg .= "$_";
    }
    #chomp($msg) if(defined $msg && $msg ne "");
    while (<ERR>)
    {
        $msg .= "$_";
    }
    close OUT;
    close ERR;
    close IN;
    waitpid $pid, 0;
    
    $code = ($?>>8);
    if($code != 0)
    {
        $code += 30;
    }
    else
    {
        $msg = "";
    }
    
    print STDERR "Add service failed($code): $msg\n" if($ctx->{debug} && $code != 0);
    
    print STDERR indent($ctx)."END: zyppServiceAdd:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;
    
    # return $code, but maybe it is ignored
    return logPrintReturn($ctx, $msg, $code); 
}

sub rugServiceAdd
{
    my $ctx = shift;
    my $url     = shift || undef;
    my $type    = shift || undef;
    my $id      = shift || undef;
    my $regcode = shift || undef;
    
    my $msg = "";

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: rugServiceAdd\n" if($ctx->{time});

    if(!$ctx->{rugzmdInstalled} || $ctx->{norug})
    {
        # If rug/zmd are not installed we are done here.
        syslog("info", "rug/zmd are not installed. finish.");
        return (0, "");
    }

    if (! defined $url || $url eq "") 
    {
        return logPrintReturn($ctx, "Missing URL.\n", 14);
    }
    if (! defined $type || $type eq "") 
    {
        $type = "zenworks";
    }

    if($ctx->{nozypp} && (lc($type) eq "zypp" || lc($type) eq "yum"))
    {
        return (0,"");
    }

    my @rugArgs = ("--quiet", "sa", "--ignore-failure", "-t", "$type");

    if(defined $regcode && $regcode ne "")
    {
        push @rugArgs, "-k", "$regcode";
    }
    push @rugArgs, "$url";
    

    if(defined $id)
    {
        push @rugArgs, $id;
    }
    else
    {
        # use url as fallback alias
        push @rugArgs, $url;
    }
        
    print STDERR "rug add service command: $ctx->{rug} ".join(" ",@rugArgs)."\n" if($ctx->{debug} >= 1);
            
    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{rug}, @rugArgs) or do {
        return logPrintReturn($ctx, "Cannot execute $ctx->{rug} ".join(" ", @rugArgs).": $!\n",13);
    };

    while (<OUT>)
    {
        $msg .= "$_";
    }
    #chomp($msg) if(defined $msg && $msg ne "");
    while (<ERR>)
    {
        $msg .= "$_";
    }
    close OUT;
    close ERR;
    close IN;
    waitpid $pid, 0;
    
    my $code = ($?>>8);
    if($code != 0)
    {
        $code += 20;
    }

    print STDERR "Add service failed($code): $msg\n" if($ctx->{debug} && $code != 0);

    print STDERR indent($ctx)."END: rugServiceAdd:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    # return $code, but maybe it is ignored
    return logPrintReturn($ctx, $msg, $code); 
}

sub rugRegister 
{
    my $ctx = shift;
    my $url     = shift || undef;
    my $regcode = shift || undef;
    my $msg = "";

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: rugRegister\n" if($ctx->{time});

    if(!$ctx->{rugzmdInstalled} || $ctx->{norug})
    {
        # If rug/zmd are not installed we are done here.
        syslog("info", "rug/zmd are not installed. finish.");
        return (0, "");
    }

    if (! defined $url || $url eq "") 
    {
        logPrintReturn($ctx, "Missing URL.\n", 14);
    }
    if (! defined $regcode || $regcode eq "") 
    {
        # no regcode; no registration required
        return (0, "");
    }

    my @rugArgs = ("--quiet", "register", "$url", "$regcode");
    
    print STDERR "rug register command: $ctx->{rug} ".join(" ",@rugArgs)."\n" if($ctx->{debug} >= 1);

    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{rug}, @rugArgs) or do {
        logPrintReturn($ctx, "Cannot execute $ctx->{rug} ".join(" ", @rugArgs).": $!\n", 13);
    };

    while (<OUT>)
    {
        $msg .= "$_";
    }
    while (<ERR>)
    {
        $msg .= "$_";
    }
    waitpid $pid, 0;
    #chomp($msg) if(defined $msg && $msg ne "");
    close IN;
    close OUT;
    close ERR;
    
    my $code = ($?>>8);
    if($code != 0)
    {
        $code += 20;
    }
        
    print STDERR "Register failed($code): $msg\n" if($ctx->{debug} && $code != 0);

    print STDERR indent($ctx)."END: rugRegister:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return logPrintReturn($ctx, $msg, $code);
}

sub rugRefresh 
{
    my $ctx = shift;

    my $code = 0;
    my $msg  = "";

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: rugRefresh\n" if($ctx->{time});

    if(!$ctx->{rugzmdInstalled} || $ctx->{norug})
    {
        # If rug/zmd are not installed we are done here.
        syslog("info", "rug/zmd are not installed. finish.");
        return (0, "");
    }

    my @rugArgs = ("--quiet", "refresh");
    
    print STDERR "rug refresh command: $ctx->{rug} ".join(" ",@rugArgs)."\n" if($ctx->{debug} >= 1);

    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{rug}, @rugArgs) or do {
        logPrintReturn($ctx, "Cannot execute $ctx->{rug} ".join(" ", @rugArgs).": $!\n", 13);
    };

    while (<OUT>)
    {
        $msg .= "$_";
    }
    while (<ERR>)
    {
        $msg .= "$_";
    }
    waitpid $pid, 0;
    #chomp($msg) if(defined $msg && $msg ne "");
    close IN;
    close OUT;
    close ERR;
    
    $code = ($?>>8);
    if($code != 0)
    {
        $code += 20;
    }
        
    print STDERR "Refresh failed($code): $msg\n" if($ctx->{debug} && $code != 0);

    print STDERR indent($ctx)."END: rugRefresh:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    # when refresh returns there is still the update-status process running
    # sleep 10 seconds to give it a chance to finish.

    sleep 10;

    return logPrintReturn($ctx, $msg, $code);
}

sub rugCatalogAdd 
{
    my $ctx  = shift;
    my $name = shift || undef;
    my $msg  = "";
    my $code = 0;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: rugCatalogAdd\n" if($ctx->{time});
    
    if(!$ctx->{rugzmdInstalled} || $ctx->{norug})
    {
        # If rug/zmd are not installed we are done here.
        syslog("info", "rug/zmd are not installed. finish.");
        return 0;
    }

    if (!defined $name) 
    {
        return logPrintReturn($ctx, "No catalogs to subscribe\n", 14);
    }

    my @rugArgs = ("--terse", "sub", "$name");
        
    print STDERR "rug subscribe command: $ctx->{rug} ".join(" ",@rugArgs)."\n" if($ctx->{debug} >= 1);
        
    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{rug}, @rugArgs) or do {
        logPrintReturn($ctx, "Cannot execute $ctx->{rug} ".join(" ", @rugArgs).": $!\n", 13);
    };
        
    while (<OUT>)
    {
        $msg .= "$_";
    }
    while (<ERR>)
    {
        $msg .= "$_";
    }
    close IN;
    close OUT;
    close ERR;
    waitpid $pid, 0;
    
    $code = ($?>>8);
    
    if($code != 0)
    {
        $code += 20;
    }
    
    $msg =~ s/^\s*//;
    $msg =~ s/\s*$//;
    
    print STDERR "Errors during subscibe($code): $msg\n" if($ctx->{debug} && $code != 0);
        
    print STDERR indent($ctx)."END: rugCatalogAdd:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;
   
    return logPrintReturn($ctx, $msg, $code);
}

sub rugCatalogDelete
{
    my $ctx  = shift;
    my $name = shift || undef;
    my $msg  = "";
    my $code = 0;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: rugCatalogDelete\n" if($ctx->{time});

    if(!$ctx->{rugzmdInstalled} || $ctx->{norug})
    {
        # If rug/zmd are not installed we are done here.
        syslog("info", "rug/zmd are not installed. finish.");
        return 0;
    }

    if (!defined $name) 
    {
        return logPrintReturn($ctx, "No catalogs to unsubscribe\n", 14);
    }

    my @rugArgs = ("--terse", "unsub", "$name");
        
    print STDERR "rug unsubscribe command: $ctx->{rug} ".join(" ",@rugArgs)."\n" if($ctx->{debug} >= 1);
        
    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{rug}, @rugArgs) or do {
        logPrintReturn($ctx, "Cannot execute $ctx->{rug} ".join(" ", @rugArgs).": $!\n", 13);
    };
        
    while (<OUT>)
    {
        $msg .= "$_";
    }
    while (<ERR>)
    {
        $msg .= "$_";
    }
    close IN;
    close OUT;
    close ERR;
    waitpid $pid, 0;
    
    $code = ($?>>8);

    if($code != 0)
    {
        $code += 20;
    }
    
    $msg =~ s/^\s*//;
    $msg =~ s/\s*$//;
    
    print STDERR "Errors during unsubscibe($code): $msg\n" if($ctx->{debug} && $code != 0);
        
    print STDERR indent($ctx)."END: rugCatalogDelete:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    # ignore errors: Very often it is "unknown catalog". 
    # We want to unsubscribe a catalog which is not available => well, no problem.
    logPrintReturn($ctx, $msg, $code);

    return (0, "");
}

sub rugServiceDelete
{
    my $ctx = shift;
    my $id  = shift || undef;
    my $type = shift || "nu";
    
    my $msg = "";

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: rugServiceDelete\n" if($ctx->{time});
    
    if(!$ctx->{rugzmdInstalled} || $ctx->{norug})
    {
        # If rug/zmd are not installed we are done here.
        syslog("info", "rug/zmd are not installed. finish.");
        return 0;
    }

    if($ctx->{nozypp} && (lc($type) eq "zypp" || lc($type) eq "yum"))
    {
        return (0,"");
    }

    if(!defined $id)
    {
        return logPrintReturn($ctx, "Missing Identifirer", 14);
    }
    
    my @rugArgs = ("--terse", "--no-abbrev", "sd", "$id");

    print STDERR "rug service delete command: $ctx->{rug} ".join(" ",@rugArgs)."\n" if($ctx->{debug} >= 1);
            
    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{rug}, @rugArgs) or do {
        return logPrintReturn($ctx, "Cannot execute $ctx->{rug} ".join(" ", @rugArgs).": $!\n",13);
    };

    while (<OUT>)
    {
        $msg .= "$_";
    }
    #chomp($msg) if(defined $msg && $msg ne "");
    while (<ERR>)
    {
        $msg .= "$_";
    }
    close OUT;
    close ERR;
    close IN;
    waitpid $pid, 0;
    
    my $code = ($?>>8);
    if($code != 0)
    {
        $code += 20;
    }

    print STDERR "Delete service failed($code): $msg\n" if($ctx->{debug} && $code != 0);

    print STDERR indent($ctx)."END: rugServiceDelete:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return logPrintReturn($ctx, $msg, $code);
}

sub zyppServiceDelete
{
    my $ctx = shift;
    my $id  = shift || undef;
    my $msg = "";
    my $code = 0;
    
    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: zyppServiceDelete\n" if($ctx->{time});

    if(!defined $id)
    {
        return logPrintReturn($ctx, "Missing Identifirer", 14);
    }
    
    if($ctx->{nozypp} || !$ctx->{zypperInstalled} || $ctx->{nozypper}) 
    {
        return (0,"");
    }
    
    # we want to delete a source. All questions are useless here.
    my @rugArgs = ("--non-interactive", "--no-gpg-checks", "sd");

    push @rugArgs, "--loose-auth", "--loose-query";

    push @rugArgs, "$id";
    
    print STDERR "zypp service delete command: $ctx->{zypper} ".join(" ",@rugArgs)."\n" if($ctx->{debug} >= 1);
    
    my $pid = open3(\*IN, \*OUT, \*ERR, $ctx->{zypper}, @rugArgs) or do {
        logPrintReturn($ctx, "Cannot execute $ctx->{zypper} ".join(" ", @rugArgs).": $!\n",13);
    };
    
    while (<OUT>)
    {
        $msg .= "$_";
    }
    #chomp($msg) if(defined $msg && $msg ne "");
    while (<ERR>)
    {
        $msg .= "$_";
    }
    close OUT;
    close ERR;
    close IN;
    waitpid $pid, 0;
    
    $code = ($?>>8);
    if($code != 0)
    {
        $code += 30;
    }
    else
    {
        $msg = "";
    }
    
    print STDERR "Delete service failed($code): $msg\n" if($ctx->{debug} && $code != 0);

    print STDERR indent($ctx)."END: zyppServiceDelete:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return logPrintReturn($ctx, $msg, $code);
}

sub rugOSTarget
{
    my $ctx = shift;
    my $msg = "";
    my $code = 1;

    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: rugOSTarget\n" if($ctx->{time});

    if(!$ctx->{rugzmdInstalled})
    {
        # If rug/zmd are not installed we are done here.
        syslog("info", "rug/zmd are not installed. finish.");
        return (0, "");
    }

    $msg = `$ctx->{rug} --terse ping`;
    $code = ($?>>8);
    
    if($code == 0)
    {
        foreach my $line (split("\n", $msg)) 
        {
            if($line =~ /^OS\s+Target:\s*(.+)/)
            {
                if(defined $1 && $1 ne "")
                {
                    $ctx->{ostarget} = $1;
                }
            }
        }
        $msg = "";
    }
    
    print STDERR indent($ctx)."END: rugOSTarget:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return ($code, $msg);
}

sub rugStart
{
    my $ctx = shift;
    my $msg = "";
    my $code = 1;
    
    $ctx->{timeindent}++;
    my $t0 = [gettimeofday] if($ctx->{time});
    print STDERR indent($ctx)."START: rugStart\n" if($ctx->{time});

    if($ctx->{rugzmdInstalled})
    {
        $msg = `$ctx->{zmdInit} status 2>&1`;
        $code = ($?>>8);
        
        if($code != 0) 
        {
            $msg = `$ctx->{zmdInit} start 2>&1`;
            $code = ($?>>8);
            
            sleep 2;
            
            foreach my $cnt (1..10)
            {
                if(-e $ctx->{GUID_FILE})
                {
                    last;
                }
                sleep 1;
            }
            
            my $msg2 = `$ctx->{zmdInit} status 2>&1`;
            $code = ($?>>8);
            
            if($code != 0) 
            {
                return logPrintReturn($ctx, "Cannot start zmd: ".($msg?$msg:"").($msg2?$msg2."($code)":"($code)"), 7);
            }
        }
    }
    else
    {
        if(!-e $ctx->{GUID_FILE})
        {
            if(!-d "/etc/zmd")
            {
                mkdir "/etc/zmd" or return logPrintReturn($ctx, "Cannot create directory /etc/zmd: $!\n", 12);
            }
            
            my $guid = `$ctx->{createGuid} 2>/dev/null`;
            if(!defined $guid || $guid eq "")
            {
                return logPrintReturn($ctx, "Cannot create guid. Command '$ctx->{createGuid}' failed.", 13);
            }
            chomp $guid;
            $guid =~ s/-//g;  # remove the -
            
            open(ZMD, "> $ctx->{GUID_FILE}") or do {
                return logPrintReturn($ctx, "Cannot open file $ctx->{GUID_FILE} for write: $!\n", 12);
            };
            print ZMD $guid;
            close ZMD;
            print STDERR "GUID created: $guid\n" if($ctx->{debug});
        }
    }

    print STDERR indent($ctx)."END: rugStart:".(tv_interval($t0))."\n" if($ctx->{time});
    $ctx->{timeindent}--;

    return (0, "");
}


sub indent
{
    my $ctx = shift;
    my $ind = "";
    for(my $i = 0;
        $i < $ctx->{timeindent};
        $i++)
    {
        $ind .= " ";
    }
    return $ind;
}

sub stripURL
{
    my $ctx = shift;
    my $url = shift || "";

    if($url eq "")
    {
        return "";
    }
    
    my $uri = URI->new($url);

    if($uri->scheme eq "http"  ||
       $uri->scheme eq "https" )
    {
        # delete user/password from url
        $uri->userinfo(undef);
    }
    
    # delete all query parameter from the url
    $uri->query(undef);
    
    return $uri->as_string;
}

sub fillURL
{
    my $ctx         = shift;
    my $url         = shift || "";
    my $queryparams = shift || undef;
    
    if($url eq "")
    {
        return "";
    }
    
    my $secret = "";
    open(SEC, "< $ctx->{SECRET_FILE}") or do {
        logPrintReturn("Cannot open file $ctx->{SECRET_FILE}: $!\n", 12);
        return "";
    };
    while(<SEC>)
    {
        $secret .= $_;
    }
    close SEC;
    
    my $uri = URI->new($url);

    if($uri->scheme eq "http"  ||
       $uri->scheme eq "https" )
    {
        # add user/password to url
        $uri->userinfo($ctx->{guid}.":$secret");
    }

    if(! defined $queryparams || $queryparams eq "")
    {
	# delete the query paramter
        $uri->query(undef);
    }
    else
    {
	# add query parameter
        $uri->query_form($queryparams);
    }
   
    return $uri->as_string;
}


1;
