/**
 * @file librtasevent_v6.h
 * @brief Structure definitions for RTAS event version 6 sections
 *
 * Copyright (C) 2005 IBM Corporation.
 * Common Public License Version 1.0 (see COPYRIGHT)
 *
 * @author Nathan Fontenot <nfont@austin.ibm.com>
 */

#ifndef _H_RE_V6_RTAS_EVENT
#define _H_RE_V6_RTAS_EVENT

#include <inttypes.h>

/**
 * @struct rtas_v6_scn_hdr
 * @brief common header for RTAS version 6 sections
 *
 * Almost every version 6 section starts with the same data as defined
 * in this structure.  This is provided to make further structure definitions 
 * and manipulation.
 */
struct rtas_v6_scn_hdr {
    char        id[2];              /**< section id */
    uint32_t    length:16;          /**< section length */
    uint32_t    version:8;          /**< section version */
    uint32_t    subtype:8;          /**< section sub-type id */
    uint32_t    creator_comp_id:16; /**< component id of section creator */
};

/* defines for the rtas_event_scn id */
#define RTAS_DUMP_SCN_ID    "DH"
#define RTAS_EPOW_SCN_ID    "EP"
#define RTAS_HMC_SCN_ID     "HM"
#define RTAS_IO_SCN_ID      "IE"
#define RTAS_IP_SCN_ID      "LP"
#define RTAS_LRI_SCN_ID     "LR"
#define RTAS_MI_SCN_ID      "MI"
#define RTAS_MTMS_SCN_ID    "MT"
#define RTAS_PSRC_SCN_ID    "PS"
#define RTAS_SSRC_SCN_ID    "SS"
#define RTAS_SW_SCN_ID      "SW"
#define RTAS_UDD_SCN_ID     "UD"

/**
 * @struct rtas_v6_main_ascn_
 * @brief RTAS version 6 Main A Section
 */
struct rtas_v6_main_a_scn {
    struct rtas_v6_scn_hdr shdr;
    
    struct rtas_date date;
    struct rtas_time time;
#if 0
    /* Date of log creation */
    union {
        uint32_t _maina_bcddate:32;
        struct date {
            uint32_t _maina_year:16;
            uint32_t _maina_month:8;
            uint32_t _maina_day:8;
        }_maina_s1;
    } _maina_u1;
#define maina_bcddate         _maina_u1._maina_bcddate
#define maina_date_year       _maina_u1._maina_s1._maina_year
#define maina_date_month      _maina_u1._maina_s1._maina_month
#define maina_date_day        _maina_u1._maina_s1._maina_day

    /* Time of log creation */
    union {
        uint32_t _miana_bcdtime:32;
        struct time {
            uint32_t _miana_hour:8;
            uint32_t _miana_minutes:8;  
            uint32_t _miana_seconds:8;
            uint32_t _miana_hundredths:8;
        } _miana_s2;
    } _miana_u2;
#define maina_bcdtime         _miana_u2._miana_bcdtime
#define maina_time_hour       _miana_u2._miana_s2._miana_hour
#define maina_time_minutes    _miana_u2._miana_s2._miana_minutes
#define maina_time_seconds    _miana_u2._miana_s2._miana_seconds
#define maina_time_hundredths _miana_u2._miana_s2._miana_hundredths

#endif
    char        platform_data[8];   /**< platform specific data */
    char	creator_id;         /**< subsystem creator id */
#define RTAS_MAINA_CREAT_SERVICE_PROC   'E'
#define RTAS_MAINA_CREAT_HYPERVISOR     'H'
#define RTAS_MAINA_CREAT_POWER_CONTROL  'W'
#define RTAS_MAINA_CREAT_PARTITION_FW   'L'

    uint32_t    /* reserved */ :16;
    uint32_t    scn_count:8;        /**< number of sections in log */

    uint32_t    /* reserved */ :32;
    char        platform_data2[8];  /**< platform spoecific data */
    uint32_t    plid:32;            /**< platform log id */
    char        platform_data3[4];  /**< platform specific data */
};

/**
 * @struct rtas_v6_main_b_scn
 * @brief RTAS version 6 Main B Section
 */
struct rtas_v6_main_b_scn {
    struct rtas_v6_scn_hdr shdr;
    
    uint32_t    subsystem_id:8;     /**< subsystem id */
    char        platform_data;      /**< platform specific data */
    uint32_t    event_type:8;       /**< error/event severity */
    uint32_t    event_subtype:8;    /**< event sub-type */
#define RTAS_MAINB_SUBTYPE_NA                   0x00
#define RTAS_MAINB_SUBTYPE_INFO_ONLY            0x01
#define RTAS_MAINB_SUBTYPE_DUMP_NOTIFICATION    0x08
#define RTAS_MAINB_SUBTYPE_PREVIOUSLY_REPORTED  0x10
#define RTAS_MAINB_SUBTYPE_DECONFIG_USER        0x20
#define RTAS_MAINB_SUBTYPE_DECONFIG_SYSTEM      0x21
#define RTAS_MAINB_SUBTYPE_RETURN_TO_NORMAL     0x30
#define RTAS_MAINB_SUBTYPE_CONCURRENT_MAINT     0x40
#define RTAS_MAINB_SUBTYPE_CAPACITY UPGRADE     0x60
#define RTAS_MAINB_SUBTYPE_RESOURCE_SPARING     0x70
#define RTAS_MAINB_SUBTYPE_DYNAMIC_RECONFIG     0x80
#define RTAS_MAINB_SUBTYPE_NORMAL_SHUTDOWN      0xD0
#define RTAS_MAINB_SUBTYPE_ABNORMAL_SHUTDOWN    0xE0
    
    char        platform_data2[4];  /**< platform specific data */
    uint32_t    /* reserved */ :16;
    uint32_t    action:16;          /**< erro action code */
#define RTAS_MAINB_ACTION_SERVICE           0x8000
#define RTAS_MAINB_ACTION_HIDDEN            0x4000
#define RTAS_MAINB_ACTION_REPORT_EXTERNALLY 0x2000
#define RTAS_MAINB_ACTION_HMC_ONLY          0x1000
#define RTAS_MAINB_ACTION_CALL_HOME         0x0800
#define RTAS_MAINB_ACTION_ISO_INCOMPLETE    0x0400

    uint32_t    /* reserved */ :32;
};

/**
 * @struct rtas_v6_ibm_platform_event
 * @brief Common definition for RTAS version 6 events
 *
 * Every RTAS version 6 event has these three items after the generic
 * RTAS headers, they are layed out in this struct to ease programming.
 */
struct rtas_v6_ibm_platform_event {
    char        pe_ibm[4];		    /**< "IBM" */
    struct rtas_v6_main_a_scn pe_main_a;
    struct rtas_v6_main_b_scn pe_main_b;
};

/**
 * @struct rtas_v6_dump_hdr
 * @brief RTAS version 6 dump locator section
 */
struct rtas_v6_dump_scn {
    struct rtas_v6_scn_hdr shdr;
    /*These defines are for the shdr.subtype field in dump sections */
#define RTAS_DUMP_SUBTYPE_FSP       0x01
#define RTAS_DUMP_SUBTYPE_PLATFORM  0x02
#define RTAS_DUMP_SUBTYPE_SMA       0x03
#define RTAS_DUMP_SUBTYPE_POWER     0x04
#define RTAS_DUMP_SUBTYPE_LOG       0x05

    uint32_t    id:32;                  /**< dump id */
    uint32_t    location:1;             /**< 0 => dump sent to HMC
                                             1 => dump sent to partition */
    uint32_t    fname_type:1;           /**< 0 => file name in ASCII
                                             1 => file name in hex */
    uint32_t    size_valid:1;           /**< dump size field valid */
    
    uint32_t    /* reserved */ :5;
    uint32_t    /* reserved */ :16;
    uint32_t    id_len:8;               /**< OS assigned dump id length */
    
    uint32_t    size_hi:32;             /**< dump size (hi-bits) */
    uint32_t    size_lo:32;             /**< dump size (low bits) */
    char        os_id[40];              /**< OS assigned dump id */
};

/**
 * @struct rtas_v6_epow_scn 
 * @brief RTAS version 6 EPOW section
 */
struct rtas_v6_epow_scn {
    struct rtas_v6_scn_hdr shdr;
 
    uint32_t    sensor_value:4;         /**< EPOW sensor value */
    uint32_t    action_code:4;          /**< EPOW action code */
#define RTAS_EPOW_ACTION_RESET              0x00
#define RTAS_EPOW_ACTION_WARN_COOLING       0x01
#define RTAS_EPOW_ACTION_WARN_POWER         0x02
#define RTAS_EPOW_ACTION_SYSTEM_SHUTDOWN    0x03
#define RTAS_EPOW_ACTION_SYSTEM_HALT        0x04
#define RTAS_EPOW_ACTION_MAIN_ENCLOSURE     0x05
#define RTAS_EPOW_ACTION_POWER_OFF          0x07

    uint32_t    event_modifier:8;       /**< EPOW event modifier */
#define RTAS_EPOW_MOD_NA                    0x00
#define RTAS_EPOW_MOD_NORMAL_SHUTDOWN       0x01
#define RTAS_EPOW_MOD_UTILITY_POWER_LOSS    0x02
#define RTAS_EPOW_MOD_CRIT_FUNC_LOSS        0x03

    uint32_t    /* reserved */ :16;
    char        reason_code[8];         /**< platform specific reason code */
};

/**
 * @struct rtas_v6_io_hdr
 * @brief RTAS version 6 i/o section
 */
struct rtas_v6_io_scn {
    struct rtas_v6_scn_hdr shdr;
   
    uint32_t    event_type:8;           /**< I/O event type */
#define RTAS_IO_TYPE_DETECTED       0x01
#define RTAS_IO_TYPE_RECOVERED      0x02
#define RTAS_IO_TYPE_EVENT          0x03
#define RTAS_IO_TYPE_RPC_PASS_THRU  0x04

    uint32_t    rpc_length:8;           /**< RPC field length.  The RPC data
                                             is optional and appears after
                                             this structure in the event if
                                             present */
    uint32_t    scope:8;                /**< event scope */
#define RTAS_IO_SCOPE_NA            0x00
#define RTAS_IO_SCOPE_RIO_HUB       0x36
#define RTAS_IO_SCOPE_RIO_BRIDGE    0x37
#define RTAS_IO_SCOPE_PHB           0x38
#define RTAS_IO_SCOPE_EADS_GLOBAL   0x39
#define RTAS_IO_SCOPE_EADS_SLOT     0x3A
    
    uint32_t    subtype:8;              /**< I/O event sub-type */
#define RTAS_IO_SUBTYPE_NA              0x00
#define RTAS_IO_SUBTYPE_REBALANCE       0x01
#define RTAS_IO_SUBTYPE_NODE_ONLINE     0x02
#define RTAS_IO_SUBTYPE_NODE_OFFLINE    0x04

    uint32_t    drc_index:32;           /**< DRC index */
};

/**
 * @struct rtas_v6_lri_hdr
 * @brief RTAS v6 logical resource identification section
 */
struct rtas_v6_lri_scn {
    struct rtas_v6_scn_hdr shdr;
    
    uint32_t    resource:8;             /**< resource type */
#define RTAS_LRI_RES_PROC           0x10
#define RTAS_LRI_RES_SHARED_PROC    0x11
#define RTAS_LRI_RES_MEM_PAGE       0x40
#define RTAS_LRI_RES_MEM_LMB        0x41

    uint32_t    /* reserved */ :8;
    uint32_t    capacity:16;            /**< entitled capacity */
    
    union {
        uint32_t _lri_cpu_id:32;        /**< logical CPU id (type = proc) */
	uint32_t _lri_drc_index:32;     /**< DRC index (type = mem LMB) */
	uint32_t _lri_mem_addr_lo;      /**< mem logical addr low bits
                                             (type = mem page) */
    } _lri_u1;
#define lri_cpu_id	_lri_u1._lri_cpu_id
#define	lri_drc_index	_lri_u1._lri_drc_index
#define lri_mem_addr_lo _lri_u1._lri_mem_addr_lo

    uint32_t    lri_mem_addr_hi:32;     /**< mem logical addr high bits
                                             (type = mem page) */
};


/**
 * @struct rtas_v6_fru_scn
 * @brief RTAS version 6 FRU callout section
 */
struct rtas_v6_fru_scn {
    uint32_t    length:8;               /**< call-out length */
    uint32_t    type:4;                 /**< callout type */
    uint32_t    subscn_included:1;      /**< fru id subsection included */
    uint32_t    other_platform_data:1;  /**< other platform data used, follows
                                             the fru subsection */
  
    uint32_t    other_platform_data2:2; /**< other platform data used, follows
                                             the fru subsection */
    char	priority;               /**< fru priority */
#define RTAS_FRU_PRIORITY_HIGH      'H'
#define RTAS_FRU_PRIORITY_MEDIUM    'M'
#define RTAS_FRU_PRIORITY_MEDIUM_A  'A'
#define RTAS_FRU_PRIORITY_MEDIUM_B  'B'
#define RTAS_FRU_PRIORITY_MEDIUM_C  'C'
#define RTAS_FRU_PRIORITY_LOW       'L'

    uint32_t    loc_code_length:8;      /**< location field length */
};
  
/**
 * @struct rtas_v6_fru_subscn
 * @brief RTAS version 6 FRU identity subsection
 *
 * Please note that the part number and procedure id fields are present
 * in this structure, if a CCIN is supplied it appears immediately after
 * this structure in the event with a length of 4.  If a FRU serial
 * number is supplied it appears after the CCIN field in the event.
 */
struct rtas_v6_fru_subscn {
    char        id[2];          /**< subsection id ("ID") */
    uint32_t    length:8;       /**< subsection length */
    uint32_t    component:4;    /**< failing component */
#define RTAS_FRU_COMP_HARDWARE          0x01
#define RTAS_FRU_COMP_CODE              0x02
#define RTAS_FRU_COMP_CONFIG_ERROR      0x03
#define RTAS_FRU_COMP_MAINT_REQUIRED    0x04
#define RTAS_FRU_COMP_EXTERNAL          0x09
#define RTAS_FRU_COMP_EXTERNAL_CODE     0x0A
#define RTAS_FRU_COMP_TOOL              0x0B
#define RTAS_FRU_COMP_SYMBOLIC          0x0C

    uint32_t    part_no:1;      /**< stocking part number provided */
    uint32_t    ccin:1;         /**< CCIN supplied */
    uint32_t    maintenance:1;  /**< Maintenance callout supplied */
    uint32_t    serial_no:1;    /**< FRU Serial number supplied */
    
    union {
	char _fru_part_no[8];
  	char _fru_procedure_id[8];
    } _fru_u1;
#define fru_part_no		_fru_u1._fru_part_no
#define fru_procedure_id	_fru_u1._fru_procedure_id
};

/**
 * @struct rtas_v6_src_hdr
 * @brief RTAS version 6 SRC section
 */
struct rtas_v6_src_scn {
    struct rtas_v6_scn_hdr shdr;

    uint32_t    version:8;          /**< SRC version */ 
    uint32_t    /* reserved */ :7;
    uint32_t    subscns:1;          /**< optional subsection is present */
    char        platform_data[6];   /**< platform specific data */

    uint32_t    ext_refcode2:32;    /**< extended reference code word 2 */
    uint32_t    ext_refcode3:32;    /**< extended reference code word 3 */
    uint32_t    ext_refcode4:32;    /**< extended reference code word 4 */
    uint32_t    ext_refcode5:32;    /**< extended reference code word 5 */

    uint32_t    ext_refcode6:32;    /**< extended reference code word 6 */
    uint32_t    ext_refcode7:32;    /**< extended reference code word 7 */
    uint32_t    ext_refcode8:32;    /**< extended reference code word 8 */
    uint32_t    ext_refcode9:32;    /**< extended reference code word 9 */

    char        primary_refcode[32];/**< primary reference code */
};

/**
 * @struct rtas_v6_src_subscn
 * @brief RTAS version 6 subsection
 */
struct rtas_v6_src_subscn {
    uint32_t    id:8;		    /**< sub-section id (0xC0) */
    uint32_t    platform_data:8;    /**< platform specific data */
    uint32_t    subscn_length:16;   /**< sub-section length */
};

/**
 * @struct rtas_v6_mt_scn
 * @brief RTAS version 6 Machine Type section
 */
struct rtas_v6_mt_scn {
    struct rtas_v6_scn_hdr shdr;
    char        model[8];           /**< machine type / model */
    char        serial_no[12];      /**< serial number */
};

#endif 

