/*
 * Copyright (C) 2008-2010 Freescale Semiconductor, Inc. All Rights Reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @file arch-mxc/mx25.h
 * @brief This file contains register definitions.
 *
 * @ingroup MSL_MX25
 */

#ifndef __ASM_ARCH_MXC_MX25_H__
#define __ASM_ARCH_MXC_MX25_H__

#ifndef __ASM_ARCH_MXC_HARDWARE_H__
#error "Do not include directly."
#endif

/*!
 * Define this option to specify we are using the newer SDMA module.
 */
#define MXC_SDMA_V2

/*
 * MX25 memory map:
 *
 * Virt     Phys        Size    What
 * ---------------------------------------------------------------------------
 * FC000000 43F00000    1M      AIPS 1
 * FC100000 50000000    1M      SPBA
 * FC200000 53F00000    1M      AIPS 2
 * FC300000 60000000    1M      ROMPATCH (128M)
 * FC400000 68000000    1M      ASIC (128M)
 * FC500000 78000000    128K    FBC RAM (IRAM)
 *          80000000    256M    SDRAM0
 *          90000000    256M    SDRAM1
 *          A0000000    128M    CS0 Flash
 *          A8000000    128M    CS1 Flash
 *          B0000000    32M     CS2 SRAM
 *          B2000000    32M     CS3
 *          B4000000    32M     CS4
 *          B6000000    32M     CS5
 * FC520000 B8000000    64K     SDRAM, WEIM, M3IF, EMI controllers
 * FC530000 BB000000    8K      NFC
 */

/*
 * IRAM
 */
#define IRAM_BASE_ADDR       0x78000000	/* internal ram */
#define IRAM_SIZE            SZ_128K

#ifdef CONFIG_SND_MXC_SOC_IRAM
#define SND_RAM_SIZE 0x10000
#else
#define SND_RAM_SIZE 0
#endif

#if defined(CONFIG_USB_STATIC_IRAM) \
    || defined(CONFIG_USB_STATIC_IRAM_PPH)
#define USB_IRAM_SIZE	SZ_8K
#else
#define USB_IRAM_SIZE 0
#endif

/*
 * AIPS 1
 */
#define AIPS1_BASE_ADDR       0x43F00000
#define AIPS1_BASE_ADDR_VIRT  0xFC000000
#define AIPS1_SIZE            SZ_1M

#define MAX_BASE_ADDR       (AIPS1_BASE_ADDR + 0x00004000)
#define CLKCTL_BASE_ADDR    (AIPS1_BASE_ADDR + 0x00008000)
#define ETB_SLOT4_BASE_ADDR (AIPS1_BASE_ADDR + 0x0000C000)
#define ETB_SLOT5_BASE_ADDR (AIPS1_BASE_ADDR + 0x00010000)
#define AAPE_BASE_ADDR      (AIPS1_BASE_ADDR + 0x00014000)
#define I2C_BASE_ADDR       (AIPS1_BASE_ADDR + 0x00080000)
#define I2C3_BASE_ADDR      (AIPS1_BASE_ADDR + 0x00084000)
#define CAN1_BASE_ADDR      (AIPS1_BASE_ADDR + 0x00088000)
#define CAN3_BASE_ADDR      (AIPS1_BASE_ADDR + 0x0008C000)
#define UART1_BASE_ADDR     (AIPS1_BASE_ADDR + 0x00090000)
#define UART2_BASE_ADDR     (AIPS1_BASE_ADDR + 0x00094000)
#define I2C2_BASE_ADDR      (AIPS1_BASE_ADDR + 0x00098000)
#define OWIRE_BASE_ADDR     (AIPS1_BASE_ADDR + 0x0009C000)
#define ATA_BASE_ADDR       (AIPS1_BASE_ADDR + 0x000A0000)
#define CSPI1_BASE_ADDR     (AIPS1_BASE_ADDR + 0x000A4000)
#define KPP_BASE_ADDR       (AIPS1_BASE_ADDR + 0x000A8000)
#define IOMUXC_BASE_ADDR    (AIPS1_BASE_ADDR + 0x000AC000)
#define AUDMUX_BASE_ADDR    (AIPS1_BASE_ADDR + 0x000B0000)
#define ECT_A_BASE_ADDR     (AIPS1_BASE_ADDR + 0x000B8000)
#define ECT_B_BASE_ADDR     (AIPS1_BASE_ADDR + 0x000BC000)

/*
 * SPBA global module enabled #0
 */
#define SPBA0_BASE_ADDR       0x50000000
#define SPBA0_BASE_ADDR_VIRT  0xFC100000
#define SPBA0_SIZE            SZ_1M

#define CSPI3_BASE_ADDR         (SPBA0_BASE_ADDR + 0x00004000)
#define UART4_BASE_ADDR         (SPBA0_BASE_ADDR + 0x00008000)
#define UART3_BASE_ADDR         (SPBA0_BASE_ADDR + 0x0000C000)
#define CSPI2_BASE_ADDR         (SPBA0_BASE_ADDR + 0x00010000)
#define SSI2_BASE_ADDR          (SPBA0_BASE_ADDR + 0x00014000)
#define ESAI_BASE_ADDR          (SPBA0_BASE_ADDR + 0x00018000)
#define ATA_DMA_BASE_ADDR       (SPBA0_BASE_ADDR + 0x00020000)
#define SIM1_BASE_ADDR          (SPBA0_BASE_ADDR + 0x00024000)
#define SIM2_BASE_ADDR          (SPBA0_BASE_ADDR + 0x00028000)
#define UART5_BASE_ADDR         (SPBA0_BASE_ADDR + 0x0002C000)
#define TSC_BASE_ADDR           (SPBA0_BASE_ADDR + 0x00030000)
#define SSI1_BASE_ADDR          (SPBA0_BASE_ADDR + 0x00034000)
#define FEC_BASE_ADDR           (SPBA0_BASE_ADDR + 0x00038000)
#define SPBA_CTRL_BASE_ADDR     (SPBA0_BASE_ADDR + 0x0003C000)

/*!
 * defines for SPBA modules
 */
#define SPBA_CSPI3   (0x1 << 2)
#define SPBA_UART4   (0x2 << 2)
#define SPBA_UART3   (0x3 << 2)
#define SPBA_CSPI2   (0x4 << 2)
#define SPBA_SSI2    (0x5 << 2)
#define SPBA_ESAI    (0x6 << 2)
#define SPBA_ATA     (0x8 << 2)
#define SPBA_SIM1    (0x9 << 2)
#define SPBA_SIM2    (0xA << 2)
#define SPBA_UART5   (0xB << 2)
#define SPBA_ANALOG  (0xC << 2)
#define SPBA_SSI1    (0xD << 2)
#define SPBA_FEC     (0xE << 2)

/*!
 * Defines for modules using static and dynamic DMA channels
 */
#define MXC_DMA_CHANNEL_IRAM         30
#define MXC_DMA_CHANNEL_UART1_RX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_UART1_TX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_UART2_RX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_UART2_TX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_UART3_RX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_UART3_TX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_UART4_RX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_UART4_TX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_UART5_RX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_UART5_TX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_MMC1         MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_SSI1_RX      MXC_DMA_DYNAMIC_CHANNEL
#ifdef CONFIG_SDMA_IRAM
#define MXC_DMA_CHANNEL_SSI1_TX      (MXC_DMA_CHANNEL_IRAM + 1)
#else
#define MXC_DMA_CHANNEL_SSI1_TX      MXC_DMA_DYNAMIC_CHANNEL
#endif
#define MXC_DMA_CHANNEL_SSI2_RX      MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_SSI2_TX      MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_CSPI1_RX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_CSPI1_TX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_CSPI2_RX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_CSPI2_TX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_CSPI3_RX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_CSPI3_TX     MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_ATA_RX       MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_ATA_TX       MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_MEMORY       MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_ESAI_RX      MXC_DMA_DYNAMIC_CHANNEL
#define MXC_DMA_CHANNEL_ESAI_TX      MXC_DMA_DYNAMIC_CHANNEL

/*
 * AIPS 2
 */
#define AIPS2_BASE_ADDR       0x53F00000
#define AIPS2_BASE_ADDR_VIRT  0xFC200000
#define AIPS2_SIZE            SZ_1M

#define CCM_BASE_ADDR       (AIPS2_BASE_ADDR + 0x00080000)
#define GPT4_BASE_ADDR      (AIPS2_BASE_ADDR + 0x00084000)
#define GPT3_BASE_ADDR      (AIPS2_BASE_ADDR + 0x00088000)
#define GPT2_BASE_ADDR      (AIPS2_BASE_ADDR + 0x0008C000)
#define GPT1_BASE_ADDR      (AIPS2_BASE_ADDR + 0x00090000)
#define EPIT1_BASE_ADDR     (AIPS2_BASE_ADDR + 0x00094000)
#define EPIT2_BASE_ADDR     (AIPS2_BASE_ADDR + 0x00098000)
#define GPIO4_BASE_ADDR     (AIPS2_BASE_ADDR + 0x0009C000)
#define PWM2_BASE_ADDR      (AIPS2_BASE_ADDR + 0x000A0000)
#define GPIO3_BASE_ADDR     (AIPS2_BASE_ADDR + 0x000A4000)
#define PWM3_BASE_ADDR      (AIPS2_BASE_ADDR + 0x000A8000)
#define SCC_BASE_ADDR       (AIPS2_BASE_ADDR + 0x000AC000)
#define RNGB_BASE_ADDR      (AIPS2_BASE_ADDR + 0x000B0000)
#define MMC_SDHC1_BASE_ADDR (AIPS2_BASE_ADDR + 0x000B4000)
#define MMC_SDHC2_BASE_ADDR (AIPS2_BASE_ADDR + 0x000B8000)
#define LCDC_BASE_ADDR      (AIPS2_BASE_ADDR + 0x000BC000)
#define SLCDC_BASE_ADDR     (AIPS2_BASE_ADDR + 0x000C0000)
#define PWM4_BASE_ADDR      (AIPS2_BASE_ADDR + 0x000C8000)
#define GPIO1_BASE_ADDR     (AIPS2_BASE_ADDR + 0x000CC000)
#define GPIO2_BASE_ADDR     (AIPS2_BASE_ADDR + 0x000D0000)
#define SDMA_BASE_ADDR      (AIPS2_BASE_ADDR + 0x000D4000)
#define WDOG1_BASE_ADDR     (AIPS2_BASE_ADDR + 0x000DC000)
#define PWM1_BASE_ADDR      (AIPS2_BASE_ADDR + 0x000E0000)
#define RTIC_BASE_ADDR      (AIPS2_BASE_ADDR + 0x000EC000)
#define IIM_BASE_ADDR       (AIPS2_BASE_ADDR + 0x000F0000)
#define USBOTG_BASE_ADDR    (AIPS2_BASE_ADDR + 0x000F4000)
#define OTG_BASE_ADDR       USBOTG_BASE_ADDR
#define CSI_BASE_ADDR       (AIPS2_BASE_ADDR + 0x000F8000)
#define DRYICE_BASE_ADDR    (AIPS2_BASE_ADDR + 0x000FC000)
#define SRTC_BASE_ADDR      (DRYICE_BASE_ADDR)

/*
 * ROMP and ASIC
 */
#define ROMP_BASE_ADDR       0x60000000
#define ROMP_BASE_ADDR_VIRT  0xFC300000
#define ROMP_SIZE            SZ_1M

#define ASIC_BASE_ADDR       0x68000000
#define ASIC_BASE_ADDR_VIRT  0xFC400000
#define ASIC_SIZE            SZ_1M
#define AVIC_BASE_ADDR       ASIC_BASE_ADDR
#define AVIC_BASE_ADDR_VIRT  ASIC_BASE_ADDR_VIRT
#define AVIC_SIZE            ASIC_SIZE

/*
 * SDRAM, WEIM, M3IF, EMI controllers
 */
#define X_MEMC_BASE_ADDR       0xB8000000
#define X_MEMC_BASE_ADDR_VIRT  0xFC520000
#define X_MEMC_SIZE            SZ_64K

#define SDRAMC_BASE_ADDR        (X_MEMC_BASE_ADDR + 0x1000)
#define WEIM_BASE_ADDR          (X_MEMC_BASE_ADDR + 0x2000)
#define M3IF_BASE_ADDR          (X_MEMC_BASE_ADDR + 0x3000)
#define EMI_CTL_BASE_ADDR       (X_MEMC_BASE_ADDR + 0x4000)

/*
 * NFC controller
 */
#define NFC_BASE_ADDR       0xBB000000
#define NFC_BASE_ADDR_VIRT  0xFC530000
#define NFC_SIZE            SZ_8K

/*
 * Memory regions and CS
 */
#define CSD0_BASE_ADDR          0x80000000
#define CSD1_BASE_ADDR          0x90000000

#define SDRAM_BASE_ADDR         CSD0_BASE_ADDR

#define CS0_BASE_ADDR           0xA0000000
#define CS1_BASE_ADDR           0xA8000000
#define CS2_BASE_ADDR           0xB0000000
#define CS3_BASE_ADDR           0xB2000000
#define CS4_BASE_ADDR           0xB4000000
#define CS4_SIZE                SZ_32M
#define CS5_BASE_ADDR           0xB6000000
#define CS5_SIZE                SZ_32M

/*!
 * This macro defines the physical to virtual address mapping for all the
 * peripheral modules. It is used by passing in the physical address as x
 * and returning the virtual address. If the physical address is not mapped,
 * it returns 0xDEADBEEF
 */
#define IO_ADDRESS(x)   \
	(void __force __iomem *) \
	(((x >= AIPS1_BASE_ADDR) && (x < (AIPS1_BASE_ADDR + AIPS1_SIZE))) ? AIPS1_IO_ADDRESS(x):\
	((x >= SPBA0_BASE_ADDR) && (x < (SPBA0_BASE_ADDR + SPBA0_SIZE))) ? SPBA0_IO_ADDRESS(x):\
	((x >= AIPS2_BASE_ADDR) && (x < (AIPS2_BASE_ADDR + AIPS2_SIZE))) ? AIPS2_IO_ADDRESS(x):\
	((x >= ROMP_BASE_ADDR) && (x < (ROMP_BASE_ADDR + ROMP_SIZE))) ? ROMP_IO_ADDRESS(x):\
	((x >= ASIC_BASE_ADDR) && (x < (ASIC_BASE_ADDR + AVIC_SIZE))) ? ASIC_IO_ADDRESS(x):\
	((x >= X_MEMC_BASE_ADDR) && (x < (X_MEMC_BASE_ADDR + X_MEMC_SIZE))) ? X_MEMC_IO_ADDRESS(x):\
	((x >= NFC_BASE_ADDR) && (x < (NFC_BASE_ADDR + NFC_SIZE))) ? NFC_IO_ADDRESS(x):\
	0xDEADBEEF)

/*
 * define the address mapping macros: in physical address order
 */

#define AIPS1_IO_ADDRESS(x)  \
	(((x) - AIPS1_BASE_ADDR) + AIPS1_BASE_ADDR_VIRT)

#define SPBA0_IO_ADDRESS(x)  \
	(((x) - SPBA0_BASE_ADDR) + SPBA0_BASE_ADDR_VIRT)

#define AIPS2_IO_ADDRESS(x)  \
	(((x) - AIPS2_BASE_ADDR) + AIPS2_BASE_ADDR_VIRT)

#define ROMP_IO_ADDRESS(x)  \
	(((x) - ROMP_BASE_ADDR) + ROMP_BASE_ADDR_VIRT)

#define ASIC_IO_ADDRESS(x)  \
	(((x) - ASIC_BASE_ADDR) + ASIC_BASE_ADDR_VIRT)

/* for entry-macro.S */
#define AVIC_IO_ADDRESS(x)	ASIC_IO_ADDRESS(x)

#define X_MEMC_IO_ADDRESS(x)  \
	(((x) - X_MEMC_BASE_ADDR) + X_MEMC_BASE_ADDR_VIRT)

#define NFC_IO_ADDRESS(x)  \
	(((x) - NFC_BASE_ADDR) + NFC_BASE_ADDR_VIRT)

#define IS_MEM_DEVICE_NONSHARED(x)	0

/*
 * DMA request assignments
 */
#define DMA_REQ_EXTREQ0    0
#define DMA_REQ_CCM        1
#define DMA_REQ_ATA_TX_END 2
#define DMA_REQ_ATA_TX     3
#define DMA_REQ_ATA_RX     4
#define DMA_REQ_CSPI2_RX   6
#define DMA_REQ_CSPI2_TX   7
#define DMA_REQ_CSPI1_RX   8
#define DMA_REQ_CSPI1_TX   9
#define DMA_REQ_UART3_RX   10
#define DMA_REQ_UART3_TX   11
#define DMA_REQ_UART4_RX   12
#define DMA_REQ_UART4_TX   13
#define DMA_REQ_EXTREQ1    14
#define DMA_REQ_EXTREQ2    15
#define DMA_REQ_UART2_RX   16
#define DMA_REQ_UART2_TX   17
#define DMA_REQ_UART1_RX   18
#define DMA_REQ_UART1_TX   19
#define DMA_REQ_SSI2_RX1   22
#define DMA_REQ_SSI2_TX1   23
#define DMA_REQ_SSI2_RX0   24
#define DMA_REQ_SSI2_TX0   25
#define DMA_REQ_SSI1_RX1   26
#define DMA_REQ_SSI1_TX1   27
#define DMA_REQ_SSI1_RX0   28
#define DMA_REQ_SSI1_TX0   29
#define DMA_REQ_NFC        30
#define DMA_REQ_ECT        31
#define DMA_REQ_ESAI_RX    32
#define DMA_REQ_ESAI_TX    33
#define DMA_REQ_CSPI3_RX   34
#define DMA_REQ_CSPI3_TX   35
#define DMA_REQ_SIM2_RX    36
#define DMA_REQ_SIM2_TX    37
#define DMA_REQ_SIM1_RX    38
#define DMA_REQ_SIM1_TX    39
#define DMA_REQ_TSC_GCQ    44
#define DMA_REQ_TSC_TCQ    45
#define DMA_REQ_UART5_RX   46
#define DMA_REQ_UART5_TX   47

/*
 *  Interrupt numbers
 */
#define MXC_INT_CSPI3               0
#define MXC_INT_GPT4                1
#define MXC_INT_OWIRE               2
#define MXC_INT_I2C                 3
#define MXC_INT_I2C2                4
#define MXC_INT_UART4               5
#define MXC_INT_RTIC                6
#define MXC_INT_ESAI                7
#define MXC_INT_SDHC2               8
#define MXC_INT_SDHC1               9
#define MXC_INT_I2C3                10
#define MXC_INT_SSI2                11
#define MXC_INT_SSI1                12
#define MXC_INT_CSPI2               13
#define MXC_INT_CSPI1               14
#define MXC_INT_ATA                 15
#define MXC_INT_GPIO3               16
#define MXC_INT_CSI                 17
#define MXC_INT_UART3               18
#define MXC_INT_IIM                 19
#define MXC_INT_SIM1                20
#define MXC_INT_SIM2                21
#define MXC_INT_RNG                 22
#define MXC_INT_GPIO4               23
#define MXC_INT_KPP                 24
#define MXC_INT_DRYICE_NORM         25
#define MXC_INT_PWM                 26
#define MXC_INT_EPIT2               27
#define MXC_INT_EPIT1               28
#define MXC_INT_GPT3                29
#define MXC_INT_POWER_FAIL          30
#define MXC_INT_CRM                 31
#define MXC_INT_UART2               32
#define MXC_INT_NANDFC              33
#define MXC_INT_SDMA                34
#define MXC_INT_USB_HTG             35
#define MXC_INT_PWM2                36
#define MXC_INT_USB_OTG             37
#define MXC_INT_SLCDC               38
#define MXC_INT_LCDC                39
#define MXC_INT_UART5               40
#define MXC_INT_PWM3                41
#define MXC_INT_PWM4                42
#define MXC_INT_CAN1                43
#define MXC_INT_CAN2                44
#define MXC_INT_UART1               45
#define MXC_INT_TSC                 46
#define MXC_INT_ECT                 48
#define MXC_INT_SCC_SCM             49
#define MXC_INT_SCC_SMN             50
#define MXC_INT_GPIO2               51
#define MXC_INT_GPIO1               52
#define MXC_INT_GPT2                53
#define MXC_INT_GPT1                54
#define MXC_INT_WDOG                55
#define MXC_INT_DRYICE_SEC          56
#define MXC_INT_FEC                 57
#define MXC_INT_EXT_INT5            58
#define MXC_INT_EXT_INT4            59
#define MXC_INT_EXT_INT3            60
#define MXC_INT_EXT_INT2            61
#define MXC_INT_EXT_INT1            62
#define MXC_INT_EXT_INT0            63

#define MXC_INT_GPT                 MXC_INT_GPT1

/* gpio and gpio based interrupt handling */
#define GPIO_DR                 0x00
#define GPIO_GDIR               0x04
#define GPIO_PSR                0x08
#define GPIO_ICR1               0x0C
#define GPIO_ICR2               0x10
#define GPIO_IMR                0x14
#define GPIO_ISR                0x18
#define GPIO_INT_LOW_LEV        0x0
#define GPIO_INT_HIGH_LEV       0x1
#define GPIO_INT_RISE_EDGE      0x2
#define GPIO_INT_FALL_EDGE      0x3
#define GPIO_INT_NONE           0x4

#define MXC_TIMER_GPT1          1
#define MXC_TIMER_GPT2          2
#define MXC_TIMER_GPT3          3
#define MXC_TIMER_GPT4          4

/*!
 * NFMS bit in RCSR register for pagesize of nandflash
 */
#define NFMS (*((volatile u32 *)IO_ADDRESS(CCM_BASE_ADDR + 0x28)))
#define NFMS_NF_DWIDTH		14
#define NFMS_NF_PG_SZ		8

#endif				/* __ASM_ARCH_MXC_MX25_H__ */
